package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import io.crnk.core.engine.information.resource.ResourceFieldType;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.jsonapi.models.ModelClassUtils;
import ru.yandex.partner.jsonapi.models.ParametrizedJavaTypeHolder;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.auth.RightHolder;

/**
 * @param <M>  Модель из core слоя
 * @param <AV> Значение в api слое
 * @param <CV> Значение в core слое
 */
public class ApiFieldBuilder<M extends Model, AV, CV> {
    /**
     * Json имя в response
     */
    private String jsonName;

    /**
     * {@link TypeReference} - удобнее использовать, так как Class обрезает generic типы
     * Class нельзя передать в виде {@literal Class<List<String>>}
     * И не обязательно передавать явно
     */
    private JavaType javaType;

    /**
     * Поле core слоя из которого получается значение api
     * В основном используется для прямого преобразования один-к-одному
     */
    private ModelPathForApi<? super M, ?> modelPath;

    /**
     * Функция получения api значения из модели core
     */
    private BatchFunction<M, AV> getterApiValue;

    /**
     * Функция получения core значения из значения api
     */
    private BatchBiFunction<M, AV> setterCoreValue;

    /**
     * Тип ресурса crnk
     * см {@link ResourceFieldType}
     */
    private ResourceFieldType resourceFieldType = ResourceFieldType.ATTRIBUTE;

    /**
     * Функция для построения {@link ApiFieldsAccessRules} для available_fields
     */
    private BiConsumer<String, ApiFieldsAccessRules.Builder<M>> availableBuilderBiConsumer;

    /**
     * Функция для построения {@link ApiFieldsAccessRules} для editable_fields
     */
    private BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> editableBuilderBiConsumer;

    /**
     * Функция для построения {@link ApiFieldsAccessRules} для add_fields
     */
    private BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> addFieldsBuilderBiConsumer;

    /**
     * Функция для получения части хеша attributes в ручке defaults
     */
    private Function<QueryParamsContext<M>, Map<String, Object>> defaultsFunction;

    /**
     * Процедура, которая получает field-specific информацию из query params запроса
     */
    private Consumer<QueryParamsContext<M>> extraQueryParamsConsumer;

    /**
     * Поля core слоя, необходимые для вычисления {@link ApiFieldsAccessRules} или значения api
     * Используется, когда для вычисления api поля нужно несколько core полей
     * Или для вычисления вспомогательных полей (available, editable и тп)
     */
    private List<ModelProperty<? super M, ?>> requiredProperties;

    /**
     * Список с json-путями, доступность которых необходимо проверить
     * Права, по которым будет вестись проверки, лежит тут {@link #innerFieldRights}
     */
    private List<List<String>> innerFieldPaths = new ArrayList<>();

    /**
     * Список прав, по которым нужно проверить json-path тут {@link #innerFieldPaths}
     */
    private List<String> innerFieldRights = new ArrayList<>();

    private String deprecated = null;

    ApiFieldBuilder(Class<AV> avClass,
                    ModelProperty<? super M, CV> modelPath,
                    BatchFunction<M, AV> getterApiValue,
                    BatchBiFunction<M, AV> setterCoreValue) {
        this(new ParametrizedJavaTypeHolder<>(ModelClassUtils.valType(avClass)), modelPath, getterApiValue,
                setterCoreValue);
    }

    ApiFieldBuilder(ParametrizedJavaTypeHolder<AV> avClass,
                    ModelProperty<? super M, CV> modelPath,
                    BatchFunction<M, AV> getterApiValue,
                    BatchBiFunction<M, AV> setterCoreValue) {
        this.javaType = avClass.getJavaType();
        this.getterApiValue = Objects.requireNonNull(getterApiValue);
        this.setterCoreValue = setterCoreValue;
        this.modelPath = ModelPath.of(modelPath);
    }

    <NV extends Model> ApiFieldBuilder(Class<AV> avClass,
                                       ModelPathForApi<? super M, CV> modelPath,
                                       BatchFunction<M, AV> getterApiValue,
                                       BatchBiFunction<M, AV> setterCoreValue) {
        this(new ParametrizedJavaTypeHolder<>(ModelClassUtils.valType(avClass)), modelPath, getterApiValue,
                setterCoreValue);
    }

    <NV extends Model> ApiFieldBuilder(ParametrizedJavaTypeHolder<AV> avClass,
                                       ModelPathForApi<? super M, CV> modelPath,
                                       BatchFunction<M, AV> getterApiValue,
                                       BatchBiFunction<M, AV> setterCoreValue) {
        this.javaType = avClass.getJavaType();
        this.getterApiValue = Objects.requireNonNull(getterApiValue);
        this.setterCoreValue = setterCoreValue;
        this.modelPath = modelPath;
    }

    public ApiFieldBuilder<M, AV, CV> withResourceFieldType(ResourceFieldType resourceFieldType) {
        this.resourceFieldType = Objects.requireNonNull(resourceFieldType);
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withAvailableFunction(
            BiConsumer<String, ApiFieldsAccessRules.Builder<M>> availableFunction) {
        if (this.availableBuilderBiConsumer != null) {
            throw new IllegalStateException("Available function is already set");
        }
        this.availableBuilderBiConsumer = availableFunction;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withEditableFunction(
            BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> editableFunction) {
        if (this.editableBuilderBiConsumer != null) {
            throw new IllegalStateException("Editable function is already set");
        }
        this.editableBuilderBiConsumer = editableFunction;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withAddFieldFunction(
            BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> addFieldFunction) {
        if (this.addFieldsBuilderBiConsumer != null) {
            throw new IllegalStateException("AddField function is already set");
        }
        this.addFieldsBuilderBiConsumer = addFieldFunction;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withAddFieldLikeEditableFunction() {
        if (this.editableBuilderBiConsumer == null) {
            throw new IllegalStateException("Editable function must be set");
        }

        if (this.addFieldsBuilderBiConsumer != null) {
            throw new IllegalStateException("AddField function is already set");
        }
        this.addFieldsBuilderBiConsumer = this.editableBuilderBiConsumer;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withAlwaysPermitAddFieldFunction() {
        if (this.addFieldsBuilderBiConsumer != null) {
            throw new IllegalStateException("AddField function is already set");
        }
        this.addFieldsBuilderBiConsumer = ApiFieldsAccessRules.alwaysPermit();
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withRequiredProperties(ModelProperty<? super M, ?> requiredProperty) {
        this.requiredProperties = List.of(requiredProperty);
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withRequiredProperties(List<ModelProperty<? super M, ?>> requiredProperties) {
        this.requiredProperties = Objects.requireNonNullElse(requiredProperties, Collections.emptyList());
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> addCheckInnerField(RightHolder right, String... jsonPath) {
        return addCheckInnerField(right.getRightName(), jsonPath);
    }

    public ApiFieldBuilder<M, AV, CV> addCheckInnerField(String rightName, String... jsonPath) {
        innerFieldPaths.add(Arrays.asList(jsonPath));
        innerFieldRights.add(rightName);
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withDefaultsFunction(
            Function<QueryParamsContext<M>, Map<String, Object>> defaultsFunction) {
        if (this.defaultsFunction != null) {
            throw new IllegalStateException("defaults function is already set");
        }
        this.defaultsFunction = defaultsFunction;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> withExtraQueryParamsConsumer(Consumer<QueryParamsContext<M>> paramsConsumer) {
        if (this.extraQueryParamsConsumer != null) {
            throw new IllegalStateException("query params consumer is already set");
        }
        this.extraQueryParamsConsumer = paramsConsumer;
        return this;
    }

    public ApiFieldBuilder<M, AV, CV> deprecated(String reason) {
        deprecated = reason;
        return this;
    }

    public ApiField<M> build(String jsonName) {
        this.jsonName = Objects.requireNonNull(jsonName);
        return new ApiField<>(this);
    }


    String getJsonName() {
        return jsonName;
    }

    JavaType getJavaType() {
        return javaType;
    }

    ModelPathForApi<? super M, ?> getModelPath() {
        return modelPath;
    }

    BatchFunction<M, AV> getGetterApiValue() {
        return getterApiValue;
    }

    BatchBiFunction<M, AV> getSetterCoreValue() {
        return setterCoreValue;
    }

    ResourceFieldType getResourceFieldType() {
        return resourceFieldType;
    }

    BiConsumer<String, ApiFieldsAccessRules.Builder<M>> getAvailableBuilderBiConsumer() {
        return availableBuilderBiConsumer;
    }

    BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> getEditableBuilderBiConsumer() {
        return editableBuilderBiConsumer;
    }

    BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<M>>> getAddFieldsBuilderBiConsumer() {
        return addFieldsBuilderBiConsumer;
    }

    Function<QueryParamsContext<M>, Map<String, Object>> getDefaultsFunction() {
        return defaultsFunction;
    }

    public Consumer<QueryParamsContext<M>> getExtraQueryParamsConsumer() {
        return extraQueryParamsConsumer;
    }

    List<ModelProperty<? super M, ?>> getRequiredProperties() {
        return requiredProperties;
    }

    List<List<String>> getInnerFieldPaths() {
        return innerFieldPaths;
    }

    List<String> getInnerFieldRights() {
        return innerFieldRights;
    }

    public String getDeprecated() {
        return deprecated;
    }
}
