package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

import ru.yandex.direct.model.Model;
import ru.yandex.partner.jsonapi.models.AvailableFunctionComposite;
import ru.yandex.partner.jsonapi.models.EditableFunctionComposite;
import ru.yandex.partner.jsonapi.models.FunctionNameEnum;
import ru.yandex.partner.libs.auth.RightHolder;

public class ApiFieldsAccessRules<T> {
    private final Set<String> always;
    private final Map<String, ApiFieldsAccessRulesFunction<T>> checkable;
    private final Map<String, Set<String>> sameAs;
    private final Map<String, String> dependsOn;
    private final Set<String> never;

    ApiFieldsAccessRules(Set<String> always,
                         Map<String, ApiFieldsAccessRulesFunction<T>> checkable,
                         Map<String, Set<String>> sameAs,
                         Set<String> never) {
        this.always = Collections.unmodifiableSet(always);
        this.checkable = Collections.unmodifiableMap(checkable);
        this.never = never;

        Map<String, String> fieldDependsOn = new HashMap<>();
        for (Map.Entry<String, Set<String>> entry : sameAs.entrySet()) {
            for (String field : entry.getValue()) {
                fieldDependsOn.put(field, entry.getKey());
            }
        }

        Map<String, Set<String>> plainSameAs = new HashMap<>();
        for (Map.Entry<String, Set<String>> entry : sameAs.entrySet()) {
            String field = entry.getKey();

            String checkableField = getCheckableFields(field, fieldDependsOn);

            plainSameAs.computeIfAbsent(checkableField, f -> new HashSet<>()).addAll(entry.getValue());

            for (String value : entry.getValue()) {
                fieldDependsOn.put(value, checkableField);
            }
        }

        this.dependsOn = Collections.unmodifiableMap(fieldDependsOn);
        this.sameAs = Collections.unmodifiableMap(plainSameAs);
    }

    private String getCheckableFields(String field, Map<String, String> dependsOn) {
        if (this.always.contains(field) || this.checkable.containsKey(field)) {
            return field;
        } else {
            return getCheckableFields(dependsOn.get(field), dependsOn);
        }
    }

    public Set<String> getAlways() {
        return always;
    }

    public Map<String, ApiFieldsAccessRulesFunction<T>> getCheckable() {
        return checkable;
    }

    public Map<String, Set<String>> getSameAs() {
        return sameAs;
    }

    public Map<String, String> getDependsOn() {
        return dependsOn;
    }

    public Set<String> getNever() {
        return this.never;
    }

    public static <T> ApiFieldsAccessRules<T> build(List<Consumer<Builder<T>>> buildList,
                                                    AvailableFunctionComposite<T> functionComposite) {
        return build(buildList,
                (Function<FunctionNameEnum, ApiFieldsAccessRulesFunction<T>>) functionComposite::getAvailableFunction);
    }

    public static <T> ApiFieldsAccessRules<T> build(List<Consumer<Builder<T>>> buildList,
                                                    EditableFunctionComposite<T> functionComposite) {
        return build(buildList,
                (Function<FunctionNameEnum, ApiFieldsAccessRulesFunction<T>>) functionComposite::getEditableFunction);
    }

    private static <T> ApiFieldsAccessRules<T> build(
            List<Consumer<Builder<T>>> buildList,
            Function<FunctionNameEnum, ApiFieldsAccessRulesFunction<T>> getter) {

        var builder = new Builder<T>();
        for (Consumer<Builder<T>> builderConsumer : buildList) {
            builderConsumer.accept(builder);
        }

        for (Map.Entry<FunctionNameEnum, String> fieldPair : builder.getFunctionToField().entrySet()) {
            var function = getter.apply(fieldPair.getKey());
            if (function == null) {
                throw new IllegalStateException("Missing function by name = " + fieldPair.getKey());
            }
            builder.checkable(fieldPair.getValue(), function);
        }

        return builder.build();
    }

    public static <T> BiConsumer<String, Builder<T>> alwaysPermit() {
        return (jsonField, builder) -> builder.always(jsonField);
    }

    public static <T> BiConsumer<String, Builder<T>> neverPermit() {
        return (jsonField, builder) -> builder.never(jsonField);
    }

    public static <T extends Model> BiConsumer<String, Builder<T>> sameAs(ApiFieldHolder<T> sameAsApiFieldHolder) {
        return (jsonField, builder) -> builder.sameAs(
                jsonField,
                sameAsApiFieldHolder.getApiField().getJsonName()
        );
    }

    public static <M extends Model, B> BiConsumer<String, Builder<B>> sameAs(ApiField<M> sameAsApiField) {
        return (jsonField, builder) -> builder.sameAs(
                jsonField,
                sameAsApiField.getJsonName()
        );
    }

    public static <T> BiConsumer<String, Builder<T>> namedFunction(FunctionNameEnum functionName) {
        return (jsonField, builder) -> builder.asFunction(jsonField, functionName);
    }

    public static <T> BiConsumer<String, Builder<T>> checkable(
            ApiFieldsAccessRulesFunction<T> checkFunction
    ) {
        return (jsonField, builder) -> builder.checkable(jsonField, checkFunction);
    }

    public static <T> BiConsumer<String, Builder<T>> checkableByRight(RightHolder rightHolder) {
        return (jsonField, builder) -> builder.checkable(
                jsonField,
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, o) -> userAuthentication.userHasRight(rightHolder))
        );
    }

    public static <T> BiConsumer<String, Builder<T>> checkableByRight(String rightName) {
        return (jsonField, builder) -> builder.checkable(
                jsonField,
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, o) -> userAuthentication.userHasRight(rightName))
        );
    }

    public static class Builder<T> {
        private Set<String> always;
        private Map<String, ApiFieldsAccessRulesFunction<T>> checkable;
        private Map<String, Set<String>> sameAs;
        private Map<FunctionNameEnum, String> functionToField;
        private Set<String> never;

        Builder() {
            always = new HashSet<>();
            checkable = new HashMap<>();
            sameAs = new HashMap<>();
            functionToField = new HashMap<>();
            never = new HashSet<>();
        }

        public Map<FunctionNameEnum, String> getFunctionToField() {
            return functionToField;
        }

        public void always(String fieldName) {
            always.add(fieldName);
        }

        public void never(String fieldName) {
            never.add(fieldName);
        }

        public void checkable(String fieldName, ApiFieldsAccessRulesFunction<T> checkFunc) {
            checkable.put(fieldName, checkFunc);
        }

        public void sameAs(String fieldName, String sameAsField) {
            sameAs.computeIfAbsent(sameAsField, s -> new HashSet<>()).add(fieldName);
        }

        private void asFunction(String fieldName, FunctionNameEnum functionName) {
            var sameFieldName = functionToField.getOrDefault(functionName, null);
            if (Objects.nonNull(sameFieldName)) {
                sameAs(
                        fieldName,
                        sameFieldName
                );
            } else {
                functionToField.put(functionName, fieldName);
            }
        }

        ApiFieldsAccessRules<T> build() {
            return new ApiFieldsAccessRules<>(always, checkable, sameAs, never);
        }
    }
}
