package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.List;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.libs.auth.model.UserAuthentication;

public class ApiFieldsAccessRulesFunction<T> {

    private final BiFunction<UserAuthentication, T, Boolean> accessRuleFunction;
    private final List<ModelProperty<? extends Model, ?>> requiredProperties;

    public ApiFieldsAccessRulesFunction(BiFunction<UserAuthentication, T, Boolean> accessRuleFunction) {
        this(accessRuleFunction, List.of());
    }

    public ApiFieldsAccessRulesFunction(BiFunction<UserAuthentication, T, Boolean> accessRuleFunction,
                                        ModelProperty<? extends Model, ?> requiredProperty) {
        this(accessRuleFunction, List.of(requiredProperty));
    }

    public ApiFieldsAccessRulesFunction(BiFunction<UserAuthentication, T, Boolean> accessRuleFunction,
                                        List<ModelProperty<? extends Model, ?>> requiredProperties) {
        this.accessRuleFunction = accessRuleFunction;
        this.requiredProperties = Objects.requireNonNullElse(requiredProperties, List.of());
    }

    public static <T> ApiFieldsAccessRulesFunction<T> alwaysAllow() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, t) -> Boolean.TRUE);
    }

    public static <T> ApiFieldsAccessRulesFunction<T> byRight(String rightName) {
        return new ApiFieldsAccessRulesFunction<>((ua, o) -> ua.userHasRight(rightName));
    }

    public BiFunction<UserAuthentication, T, Boolean> getAccessRuleFunction() {
        return accessRuleFunction;
    }

    public List<ModelProperty<? extends Model, ?>> getRequiredProperties() {
        return requiredProperties;
    }

    public static <M> ApiFieldsAccessRulesFunction<M> and(
            List<ApiFieldsAccessRulesFunction<M>> functions
    ) {
        var function = new BiFunction<UserAuthentication, M, Boolean>() {
            @Override
            public Boolean apply(UserAuthentication userAuthentication, M model) {
                for (ApiFieldsAccessRulesFunction<M> function : functions) {
                    if (!function.getAccessRuleFunction().apply(userAuthentication, model)) {
                        return false;
                    }
                }
                return true;
            }
        };
        return new ApiFieldsAccessRulesFunction<>(function,
                functions.stream()
                        .map(ApiFieldsAccessRulesFunction::getRequiredProperties)
                        .flatMap(List::stream)
                        .collect(Collectors.toList())
        );
    }

    public static <M> BiConsumer<String, ApiFieldsAccessRules.Builder<M>> checkableAnd(
            List<ApiFieldsAccessRulesFunction<M>> functions
    ) {
        return ApiFieldsAccessRules.checkable(and(functions));
    }

    public static <M> ApiFieldsAccessRulesFunction<M> or(
            List<ApiFieldsAccessRulesFunction<M>> functions
    ) {
        var function = new BiFunction<UserAuthentication, M, Boolean>() {
            @Override
            public Boolean apply(UserAuthentication userAuthentication, M model) {
                for (ApiFieldsAccessRulesFunction<M> function : functions) {
                    if (function.getAccessRuleFunction().apply(userAuthentication, model)) {
                        return true;
                    }
                }
                return false;
            }
        };

        return new ApiFieldsAccessRulesFunction<>(function,
                functions.stream()
                        .map(ApiFieldsAccessRulesFunction::getRequiredProperties)
                        .flatMap(List::stream)
                        .collect(Collectors.toList())
        );
    }

    public static <T> BiConsumer<String, ApiFieldsAccessRules.Builder<T>> checkableOr(
            List<ApiFieldsAccessRulesFunction<T>> functions
    ) {
        return ApiFieldsAccessRules.checkable(or(functions));
    }
}
