package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.libs.auth.model.UserAuthentication;

public class ApiFieldsAccessRulesServiceImpl<T> implements ApiFieldsAccessRulesService<T> {
    private final ApiFieldsAccessRules<T> apiFieldsAccessRules;
    private final List<ModelProperty<? extends Model, ?>> requiredModelProperties;

    public ApiFieldsAccessRulesServiceImpl(ApiFieldsAccessRules<T> apiFieldsAccessRules) {
        this.apiFieldsAccessRules = apiFieldsAccessRules;
        this.requiredModelProperties = apiFieldsAccessRules.getCheckable().values().stream()
                .map(ApiFieldsAccessRulesFunction::getRequiredProperties)
                .flatMap(List::stream)
                .distinct()
                .collect(Collectors.toList());
    }

    @Override
    public Set<String> calculate(UserAuthentication userAuthentication, T entity) {
        var result = new HashSet<>(apiFieldsAccessRules.getAlways());

        for (var entry : apiFieldsAccessRules.getCheckable().entrySet()) {
            if (entry.getValue().getAccessRuleFunction().apply(userAuthentication, entity)) {
                result.add(entry.getKey());
            }
        }

        for (var entry : apiFieldsAccessRules.getSameAs().entrySet()) {
            if (result.contains(entry.getKey())) {
                result.addAll(entry.getValue());
            }
        }

        var never = apiFieldsAccessRules.getNever();

        return result.stream().filter(it -> !never.contains(it)).collect(Collectors.toSet());
    }

    @Override
    public Set<String> calculate(UserAuthentication userAuthentication, T entity, IncomingApiFields incomingApiFields) {
        var result = new HashSet<>(apiFieldsAccessRules.getAlways());

        Set<String> updatedFields =
                incomingApiFields.getUpdatedFields();

        Map<String, ApiFieldsAccessRulesFunction<T>> checks = apiFieldsAccessRules.getCheckable();

        for (var entry : checks.entrySet()) {
            if (!updatedFields.contains(entry.getKey())) {
                continue;
            }

            if (entry.getValue().getAccessRuleFunction().apply(userAuthentication, entity)) {
                result.add(entry.getKey());
            }
        }

        for (var entry : apiFieldsAccessRules.getSameAs().entrySet()) {
            var sameAsFields = entry.getValue();

            var dependentFields = sameAsFields.stream()
                    .filter(f -> updatedFields.contains(f))
                    .collect(Collectors.toSet());

            if (dependentFields.isEmpty()) {
                continue;
            }

            String fieldWithCheck = entry.getKey();

            ApiFieldsAccessRulesFunction<T> check = checks.get(fieldWithCheck);

            Boolean checkResult;
            if (Objects.nonNull(check)) {
                checkResult = check.getAccessRuleFunction().apply(userAuthentication, entity);
            } else {
                checkResult = result.contains(fieldWithCheck);
            }

            if (checkResult) {
                result.addAll(dependentFields);
            }
        }

        return result;
    }

    @Override
    public List<ModelProperty<? extends Model, ?>> getRequiredModelProperties() {
        return requiredModelProperties;
    }

    @Override
    public Set<ModelProperty<? extends Model, ?>> getRequiredModelPropertiesByJsonName(String jsonName) {
        if (apiFieldsAccessRules.getAlways().contains(jsonName)) {
            return Collections.emptySet();
        }

        String searchBy = apiFieldsAccessRules.getDependsOn().getOrDefault(jsonName, jsonName);

        Map<String, ApiFieldsAccessRulesFunction<T>> checks = apiFieldsAccessRules.getCheckable();

        if (checks.containsKey(searchBy)) {
            return new HashSet<>(checks.get(searchBy).getRequiredProperties());
        } else {
            return Collections.emptySet();
        }
    }

    @Override
    public Set<String> getNever() {
        return apiFieldsAccessRules.getNever();
    }
}
