package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.MapUtils;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.jsonapi.models.AccessFunctionComposite;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;

public abstract class ApiFieldsService<M extends ModelWithId> {

    private final ApiFieldsAccessRulesService<M> apiFieldsAvailableRules;
    private final ApiFieldsAccessRulesService<EditableData<M>> apiAddFieldsRules;
    private final ApiFieldsAccessRulesService<EditableData<M>> apiFieldsEditableRules;
    private final Collection<ApiField<M>> apiFields;
    private final Map<String, ApiField<M>> apiFieldMap;
    private final Map<String, RequiredModelProperties> requiredModelPropertiesByJsonName;
    private final List<CheckInnerField> checkInnerFields;
    private final String availableFieldJsonName;
    private final ApiModel<M> apiModel;
    private final List<ForeignFieldDefaultsService<M>> foreignFieldDefaultsServices;


    public ApiFieldsService(ApiFieldsAccessRulesService<M> apiFieldsAvailableRules,
                            ApiFieldsAccessRulesService<EditableData<M>> apiFieldsEditableRules,
                            ApiModel<M> apiModel,
                            String availableFieldJsonName,
                            String editableFieldJsonName,
                            AccessFunctionComposite<M> functionComposite,
                            List<ForeignFieldDefaultsService<M>> foreignFieldDefaultsServices) {
        this.apiFieldsAvailableRules = apiFieldsAvailableRules;
        this.apiFieldsEditableRules = apiFieldsEditableRules;
        this.apiFields = apiModel.getFields();
        this.apiModel = apiModel;
        this.availableFieldJsonName = availableFieldJsonName;
        this.foreignFieldDefaultsServices = foreignFieldDefaultsServices;

        apiFieldMap = apiFields.stream()
                .collect(Collectors.toMap(ApiField::getJsonName, Function.identity()));

        try {
            apiAddFieldsRules = new ApiFieldsAccessRulesServiceImpl<>(
                    ApiFieldsAccessRules.build(
                            apiFields.stream()
                                    .map(ApiField::getAddFieldsFunction)
                                    .filter(Objects::nonNull)
                                    .collect(Collectors.toList()),
                            functionComposite)
            );
        } catch (IllegalStateException e) {
            throw new IllegalStateException(
                    e.getMessage() + ". AddFieldFunction for Model = " + apiModel.getResourceType());
        }

        requiredModelPropertiesByJsonName =
                calculateRequiredModelPropertiesByJsonName(apiFields,
                        availableFieldJsonName, this.apiFieldsAvailableRules, editableFieldJsonName,
                        this.apiFieldsEditableRules);

        checkInnerFields = apiFields.stream()
                .flatMap(apiField -> apiField.getCheckInnerFields().stream())
                .collect(Collectors.toList());
    }

    public ApiFieldsAccessRulesService<EditableData<M>> getApiFieldsEditPermitRules() {
        return apiFieldsEditableRules;
    }

    public Set<ModelProperty<? extends Model, ?>> modelPropertiesFromRequestedFieldNames(
            Set<String> requestedFieldNames
    ) {
        return requestedFieldNames.stream()
                .map(this::getRequiredModelPropertiesForReadByJsonName)
                .flatMap(Collection::stream).collect(Collectors.toSet());
    }


    public Collection<ApiField<M>> getApiFields() {
        return apiFields;
    }

    public List<ApiField<M>> getPermittedApiFields() {
        var never = apiFieldsAvailableRules.getNever();
        return apiFields.stream().filter(it -> !never.contains(it.getJsonName())).collect(Collectors.toList());
    }

    public List<ApiField<M>> permittedApiFields(UserAuthentication userAuthentication) {
        var permitted = apiFieldsAvailableRules.calculate(userAuthentication, apiModel.createNewInstance());
        return apiFields.stream().filter(it -> permitted.contains(it.getJsonName())).collect(Collectors.toList());
    }

    public Set<ModelProperty<? extends Model, ?>> getRequiredModelPropertiesForReadByJsonName(String jsonName) {
        RequiredModelProperties requiredModelProperties = requiredModelPropertiesByJsonName.getOrDefault(jsonName,
                new RequiredModelProperties());

        Set<ModelProperty<? extends Model, ?>> requiredFields =
                new HashSet<>(requiredModelProperties.getRequiredFields());
        requiredFields.addAll(requiredModelProperties.getAvailableFields());

        return requiredFields;
    }

    public RequiredModelProperties getRequiredModelPropertiesByJsonName(String jsonName) {
        return requiredModelPropertiesByJsonName.getOrDefault(jsonName, new RequiredModelProperties());
    }

    public abstract ModelProperty<? super M, ?> getId();

    public Map<String, JsonNode> filter(AuthenticationFacade authenticationFacade, Map<String, JsonNode> attributes) {
        if (MapUtils.isEmpty(attributes)) {
            return attributes;
        }

        for (CheckInnerField checkInnerField : checkInnerFields) {
            String rootNode = checkInnerField.getRootFieldName();
            // проверка может уже вырезали этот объект
            if (!attributes.containsKey(rootNode)) {
                continue;
            }

            if (authenticationFacade.userHasRight(checkInnerField.getRight())) {
                continue;
            }

            if (checkInnerField.getJsonPath().isEmpty()) {
                attributes.remove(rootNode);
            } else {
                removeInnerJsonNode(attributes.get(rootNode), checkInnerField.getJsonPath());
            }
        }

        return attributes;
    }

    private void removeInnerJsonNode(JsonNode jsonNode, List<String> jsonPath) {
        var jsonNodes = new ArrayList<JsonNode>(jsonPath.size());
        for (String fieldName : jsonPath) {
            if (jsonNode == null) {
                break;
            }
            // добавляем JsonNode, который должен иметь в себе fieldName
            jsonNodes.add(jsonNode);
            jsonNode = jsonNode.get(fieldName);
        }

        // если размер списка разный значит мы не смогли пройти полный путь jsonPath
        // если размер одинаковый то значит последний элемент jsonNodes это объект,
        // который должен хранить в себе искомый удаляемый атрибут (последний элемент в списке jsonPath)
        if (jsonNodes.size() == jsonPath.size()) {
            int lastIndex = jsonNodes.size() - 1;
            JsonNode node = jsonNodes.get(lastIndex);
            String fieldName = jsonPath.get(lastIndex);

            if (node.isObject()) {
                ((ObjectNode) node).remove(fieldName);
            }
        }
    }

    private Map<String, RequiredModelProperties> calculateRequiredModelPropertiesByJsonName(
            Collection<ApiField<M>> apiFields,
            String availableFieldJsonName,
            ApiFieldsAccessRulesService<M> apiFieldsAvailableRules,
            String editableFieldJsonName,
            ApiFieldsAccessRulesService<EditableData<M>> apiFieldsEditableRules
    ) {
        Map<String, RequiredModelProperties> map = Maps.newHashMapWithExpectedSize(apiFields.size() + 2);

        for (ApiField<M> apiField : apiFields) {
            String jsonName = apiField.getJsonName();

            RequiredModelProperties requiredModelProperties = new RequiredModelProperties()
                    .putRequiredFields(apiField.getRequiredProperties().stream()
                            .map(field -> (ModelProperty<? extends Model, ?>) field).collect(Collectors.toSet())
                    )
                    .putAvailableFields(apiFieldsAvailableRules.getRequiredModelPropertiesByJsonName(jsonName))
                    .putEditableFields(apiFieldsEditableRules.getRequiredModelPropertiesByJsonName(jsonName))
                    .putAddFields(apiAddFieldsRules.getRequiredModelPropertiesByJsonName(jsonName));

            ModelPathForApi<? super M, ?> modelPathForApi = apiField.getModelPath();
            if (modelPathForApi != null) {
                requiredModelProperties.putRequiredFields(Set.of(modelPathForApi.rootProperty()));
            }

            map.put(jsonName, requiredModelProperties);
        }

        map.computeIfAbsent(availableFieldJsonName, f -> new RequiredModelProperties())
                .putRequiredFields(new HashSet<>(apiFieldsAvailableRules.getRequiredModelProperties()));

        map.computeIfAbsent(editableFieldJsonName, f -> new RequiredModelProperties())
                .putRequiredFields(new HashSet<>(apiFieldsEditableRules.getRequiredModelProperties()));

        return Collections.unmodifiableMap(map);
    }

    public ApiModel<M> getApiModel() {
        return apiModel;
    }

    public Map<String, JsonNode> convertToApi(M model,
                                              Set<String> fields,
                                              ObjectMapper objectMapper,
                                              AuthenticationFacade authenticationFacade) {
        var map = Maps.<String, JsonNode>newHashMapWithExpectedSize(fields.size());
        if (fields.isEmpty()) {
            return map;
        }

        Map<String, Boolean> availableFields;
        var apiAvailableFieldsField = apiFieldMap.getOrDefault(availableFieldJsonName, null);
        if (apiAvailableFieldsField != null) {
            availableFields = (Map<String, Boolean>) apiAvailableFieldsField.toApiValue(model);
        } else {
            availableFields = apiFieldsAvailableRules.calculate(authenticationFacade.getUserAuthentication(), model)
                    .stream()
                    .map(it -> Map.entry(it, true))
                    .collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));
        }

        var fieldsToReturn = fields
                .stream()
                .filter(availableFields::containsKey)
                .collect(Collectors.toSet());

        for (String field : fieldsToReturn) {
            var apiField = apiFieldMap.get(field);
            var node = map.getOrDefault(apiField.getJsonName(), objectMapper.valueToTree(apiField.toApiValue(model)));
            map.put(apiField.getJsonName(), node);
        }

        return filter(authenticationFacade, map);
    }

    public Set<String> getAddFields(AuthenticationFacade authenticationFacade, M model) {
        return apiAddFieldsRules.calculate(authenticationFacade.getUserAuthentication(), new EditableData<>(model));
    }

    public Map<String, Object> getDefaults(QueryParamsContext<M> queryParamsContext, Set<String> fieldNames) {
        Map<String, Object> result = new HashMap<>();
        for (var apiField : apiFields) {
            for (String fieldName : fieldNames) {
                if (apiField.getJsonName().equals(fieldName)) {
                    apiField.getDefaults(queryParamsContext).forEach((k, v) -> {
                        if (v != null) {
                            result.merge(k, v, (v1, v2) -> {
                                        if (v2 instanceof Map<?, ?> && v1 instanceof Map<?, ?>) {
                                            return Stream.of((Map<String, Object>) v2, (Map<String, Object>) v1)
                                                    .flatMap(m -> m.entrySet().stream())
                                                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                                        }
                                        return v2;
                                    }
                            );
                        } else {
                            result.put(k, v);
                        }
                    });
                }
            }
        }

        for (var foreignDefaultsService : foreignFieldDefaultsServices) {
            for (String fieldName : fieldNames) {
                if (foreignDefaultsService.getJsonName().equals(fieldName)) {
                    result.putAll(foreignDefaultsService.getDefaults(queryParamsContext));
                }
            }
        }

        return result;
    }

    public void consumeExtraQueryParams(QueryParamsContext<M> queryParamsContext) {
        apiFields.forEach(apiField -> apiField.consumeExtraQueryParams(queryParamsContext));
    }

    public abstract Set<ModelProperty<?, ?>> getPropertiesForValidate(Set<ModelProperty<?, ?>> affectedFields);
}
