package ru.yandex.partner.jsonapi.crnk.fields;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

public class PropertyChainPath<M extends Model, CM extends Model, V> implements ModelPathForApi<M, V> {
    private static final ModelPath ROOT_PATH = new ModelPath<>() {
        @Override
        public Object get(Model model) {
            return model;
        }

        @Override
        public void set(Model model, Object value) {
            throw new UnsupportedOperationException("Can't call setter on ROOT_PATH");
        }

        @Override
        public Object getOrCreate(Model model, Supplier<Object> inst) {
            if (model == null) {
                throw new UnsupportedOperationException("Can't create on ROOT_PATH");
            }
            return model;
        }
    };

    private final ModelPath<M, CM> pathToHere;
    private final ModelProperty<CM, V> property;
    private final Supplier<CM> instantiator;

    public PropertyChainPath(ModelProperty<CM, V> currentProp) {
        this(
                (ModelPath<M, CM>) ROOT_PATH,
                currentProp,
                defaultInstantiator(currentProp)
        );
    }

    public PropertyChainPath(ModelPath<M, CM> pathToHere, ModelProperty<CM, V> property, Supplier<CM> instantiator) {
        this.pathToHere = pathToHere;
        this.property = property;
        this.instantiator = instantiator;
    }

    static <M extends Model> Supplier<M> defaultInstantiator(ModelProperty<M, ?> currentProp) {
        return () -> {
            try {
                return currentProp.getModelClass()
                        .getDeclaredConstructor().newInstance();
            } catch (Exception e) {
                throw new RuntimeException("No constructor found for default instantiation strategy!", e);
            }
        };
    }

    @Override
    public V get(M model) {
        return Optional.of(model)
                .map(pathToHere::get)
                .map(property::get)
                .orElse(null);
    }

    @Override
    public void set(M model, V value) {
        Objects.requireNonNull(model, "Can't set value for empty model");

        property.set(pathToHere.getOrCreate(model, instantiator), value);
    }

    @Override
    public V getOrCreate(M model, Supplier<V> inst) {
        var curModel = pathToHere.getOrCreate(model, instantiator);

        var propValue = property.get(curModel);
        if (propValue == null) {
            propValue = inst.get();
            property.set(curModel, propValue);
        }

        return propValue;
    }

    @Override
    public ModelProperty<M, ?> rootProperty() {
        PropertyChainPath<M, ?, ?> curPath = this;
        while (curPath.pathToHere != ROOT_PATH) {
            curPath = (PropertyChainPath<M, ?, ?>) curPath.pathToHere;
        }
        return (ModelProperty<M, ?>) curPath.property;
    }

    @Override
    public ModelProperty<?, V> lastProperty() {
        return property;
    }

    @Override
    public boolean isNested() {
        return this.pathToHere != ROOT_PATH;
    }
}
