package ru.yandex.partner.jsonapi.crnk.filter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.partner.core.filter.CoreFilter;
import ru.yandex.partner.core.filter.CoreFilterWithValues;
import ru.yandex.partner.core.filter.FilterType;
import ru.yandex.partner.core.filter.exceptions.FilterIllegalValuesException;
import ru.yandex.partner.core.filter.exceptions.FilterOperatorException;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.jsonapi.crnk.filter.description.provider.DictionaryValuesProvider;
import ru.yandex.partner.jsonapi.crnk.filter.expose.CrnkFilterExposeStrategy;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterParserException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.IgnorableFilterParserException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.CrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.DefaultCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.FlexibleBooleanCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.NumberFilterValueParser;
import ru.yandex.partner.jsonapi.messages.FilterMsg;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

import static java.util.Objects.requireNonNull;

public abstract class CrnkFilter<M, V> {
    private final String name;
    private final String group;
    private final MetaFilter<? super M, V> metaFilter;
    private final CrnkFilterValueParser<V> crnkFilterValueParser;

    private static final Map<Class<?>, CrnkFilterValueParser<?>> DEFAULT_PARSERS = Map.of(
            Boolean.class, new FlexibleBooleanCrnkFilterValueParser(),
            Long.class, new NumberFilterValueParser()
    );

    public CrnkFilter(String name,
                      String group,
                      MetaFilter<? super M, V> metaFilter,
                      CrnkFilterValueParser<V> crnkFilterValueParser
    ) {
        this.name = name;
        this.group = group;
        this.metaFilter = metaFilter;
        this.crnkFilterValueParser = crnkFilterValueParser == null
                ? getDefaultValueParser(metaFilter.getValueClass())
                : crnkFilterValueParser;
    }

    public String getName() {
        return name;
    }

    public String getGroup() {
        return group;
    }

    public FilterType getFilterType() {
        return metaFilter.getFilterType();
    }

    public abstract DictionaryValuesProvider<?> getDictionaryValuesProvider();

    public abstract List<CrnkFilterEntity> getCrnkFilterEntities();

    public CoreFilter<M> toCoreFilter(FilterOperator filterOperator, JsonNode jsonNode)
            throws IgnorableFilterParserException {
        var values = crnkFilterValueParser.toValues(jsonNode);
        try {
            metaFilter.getFilterType().check(filterOperator, values);
        } catch (FilterOperatorException e) {
            throw new CrnkFilterParserException(
                    MsgWithArgs.of(
                            FilterMsg.BAD_OPERATOR_IN_FILTER,
                            filterOperator,
                            name,
                            metaFilter.getFilterType().getText())
            );
        } catch (FilterIllegalValuesException e) {
            throw new CrnkFilterParserException(
                    MsgWithArgs.of(
                            FilterMsg.INCORRECT_DATA_FOR_FILTER,
                            name,
                            metaFilter.getFilterType().getText()
                    )
            );
        }
        return new CoreFilterWithValues<>(metaFilter, filterOperator, values);
    }

    private CrnkFilterValueParser<V> getDefaultValueParser(Class<V> valueClass) {
        if (DEFAULT_PARSERS.containsKey(valueClass)) {
            //noinspection unchecked Мы правильно складываем ручками в DEFAULT_PARSERS
            return (CrnkFilterValueParser<V>) DEFAULT_PARSERS.get(valueClass);
        }
        return new DefaultCrnkFilterValueParser<>(valueClass);
    }

    public CrnkFilterValueParser<V> getCrnkFilterValueParser() {
        return crnkFilterValueParser;
    }

    /**
     * Чтобы в апи (например в поле meta) использовалось вложенное имя для фильтра через точку,
     * нужно передать **несколько** CrnkFilterName.
     */
    public static CrnkFilterMatchBuilder builder(CrnkFilterName crnkFilterName, CrnkFilterName... crnkFilterNames) {
        return new CrnkFilterMatchBuilder(crnkFilterName, crnkFilterNames);
    }

    public static class CrnkFilterMatchBuilder extends CrnkFilterBuilder {
        private CrnkFilterMatchBuilder(CrnkFilterName crnkFilterName, CrnkFilterName... crnkFilterNames) {
            super(crnkFilterName, crnkFilterNames);
        }

        public <M, V> CrnkFilterMatch<M, V> toCrnkFilterMatch(
                MetaFilter<? super M, V> metaFilter,
                CrnkFilterValueParser<V> crnkFilterValueParser,
                CrnkFilterBuilder... filterEntities
        ) {
            return new CrnkFilterMatch<>(
                    getName(),
                    getGroup(),
                    metaFilter,
                    crnkFilterValueParser,
                    filterEntities
            );
        }
    }

    public static class CrnkFilterBuilder {
        private List<CrnkFilterName> crnkFilterNames;
        private String group;
        private GettextMsg label;
        private GettextMsg hint;
        private CrnkFilterExposeStrategy exposeStrategy = () -> false;

        private CrnkFilterBuilder(CrnkFilterName name, CrnkFilterName... crnkFilterNames) {
            this.crnkFilterNames = new ArrayList<>();
            this.crnkFilterNames.add(name);
            this.crnkFilterNames.addAll(Arrays.asList(crnkFilterNames));
        }

        public CrnkFilterBuilder setGroup(String group) {
            this.group = group;
            return this;
        }

        public CrnkFilterBuilder setLabel(GettextMsg label) {
            this.label = label;
            return this;
        }

        public CrnkFilterBuilder setHint(GettextMsg hint) {
            this.hint = hint;
            return this;
        }

        public CrnkFilterBuilder setExposed(@Nonnull CrnkFilterExposeStrategy exposeStrategy) {
            requireNonNull(exposeStrategy);
            this.exposeStrategy = exposeStrategy;
            return this;
        }

        public List<CrnkFilterName> getCrnkFilterNames() {
            return crnkFilterNames;
        }

        public String getName() {
            return crnkFilterNames.get(0).getName();
        }

        public String getGroup() {
            return group;
        }

        public GettextMsg getLabel() {
            return label;
        }

        public GettextMsg getHint() {
            return hint;
        }

        public CrnkFilterExposeStrategy getExposeStrategy() {
            return exposeStrategy;
        }

        public <M, V> CrnkFilterSimple<M, V> toCrnkFilter(MetaFilter<? super M, V> metaFilter) {
            return toCrnkFilter(metaFilter, null);
        }

        public <M, V extends Comparable<V>> CrnkFilterSimple<M, V> toCrnkFilter(
                MetaFilter<? super M, V> metaFilter,
                DictionaryValuesProvider<V> dictionaryValuesProvider
        ) {
            return toCrnkFilter(metaFilter, null, dictionaryValuesProvider);
        }

        public <M, V> CrnkFilterSimple<M, V> toCrnkFilter(
                MetaFilter<? super M, V> metaFilter,
                CrnkFilterValueParser<V> crnkFilterValueParser
        ) {
            return new CrnkFilterSimple<>(
                    this,
                    metaFilter,
                    null,
                    crnkFilterValueParser
            );
        }

        public <M, V extends Comparable<V>> CrnkFilterSimple<M, V> toCrnkFilter(
                MetaFilter<? super M, V> metaFilter,
                CrnkFilterValueParser<V> crnkFilterValueParser,
                DictionaryValuesProvider<?> dictionaryValuesProvider
        ) {
            return new CrnkFilterSimple<>(
                    this,
                    metaFilter,
                    dictionaryValuesProvider,
                    crnkFilterValueParser
            );
        }
    }

    @Override
    public String toString() {
        return "CrnkFilter{" +
                "name='" + name + '\'' +
                ", group='" + group + '\'' +
                ", metaFilter=" + metaFilter +
                '}';
    }
}
