package ru.yandex.partner.jsonapi.crnk.filter.parser;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterIncorrectException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterParserException;
import ru.yandex.partner.jsonapi.messages.ControllerMsg;

import static ru.yandex.partner.core.filter.operator.FilterOperator.MATCH;

@Service
public class CrnkFilterParser {
    private final ObjectMapper objectMapper;

    public CrnkFilterParser(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    public FilterNode parse(String json) {
        try {
            return parse(objectMapper.readValue(json, JsonNode.class));
        } catch (IOException e) {
            throw new CrnkFilterParserException(e, ControllerMsg.PARAMETER_FILTER_INCORRECT);
        }
    }

    public FilterNode parse(JsonNode jsonNode) {
        return toFilterNode(jsonNode, BinaryOperator.AND);
    }

    private FilterNode toFilterNode(JsonNode node, BinaryOperator contextOperator) {
        if (node.isArray()) {
            return toFilterNode((ArrayNode) node, contextOperator);
        } else if (node.isObject()) {
            return toFilterNode((ObjectNode) node, contextOperator);
        } else {
            throw new CrnkFilterParserException(ControllerMsg.SUBFILTERS_USE_ARRAY_OR_SCALAR);
        }
    }

    private FilterNode toFilterNode(ArrayNode arrayNode, BinaryOperator contextOperator) {
        int size = arrayNode.size();

        if (size == 0) {
            throw new CrnkFilterIncorrectException("Filter is empty");
        }

        if (size < 2) {
            throw new CrnkFilterIncorrectException("Unexpected Node: " + arrayNode);
        }
        JsonNode firstNode = arrayNode.get(0);
        JsonNode secondNode = arrayNode.get(1);
        if (size == 3 && firstNode.isTextual() && secondNode.isTextual()) {
            // arrayNode имеет вид:
            // ["filter-name", "LIKE", "value"], второй элемент это оператор фильтрации
            // Конечная единица
            String operatorText = secondNode.textValue();
            FilterOperator filterOperator = getVerifiedFilterOperator(operatorText);
            String filterName = firstNode.textValue();
            JsonNode valueNode = arrayNode.get(2);
            return new FilterNode(createValueFilter(filterName, filterOperator, valueNode));
        } else if (size == 2 && firstNode.isTextual() && secondNode.isArray()) {
            // ["AND", [[], []]]
            BinaryOperator binaryOperator = getVerifiedBinaryOperator(firstNode.textValue());
            // secondNode = [[], [], ..]
            List<FilterNode> filterNodes = StreamSupport.stream(secondNode.spliterator(), false)
                    .map(node -> {
                        JsonNode unwrappedNode = node;
                        while (unwrappedNode.isArray() && unwrappedNode.size() == 1) {
                            // just unwrap :
                            // [[]] OR [{}]
                            unwrappedNode = unwrappedNode.get(0);
                        }
                        return toFilterNode(unwrappedNode, binaryOperator);
                    })
                    .collect(Collectors.toList());

            return new FilterNode(binaryOperator, filterNodes);
        } else {
            throw new CrnkFilterIncorrectException("Unexpected Node: " + arrayNode);
        }
    }

    private FilterNode toFilterNode(ObjectNode objectNode, BinaryOperator contextOperator) {
        Iterator<String> fieldNameIterator = objectNode.fieldNames();
        var filterNodes = new ArrayList<FilterNode>();
        while (fieldNameIterator.hasNext()) {
            String filterName = fieldNameIterator.next();
            JsonNode valueNode = objectNode.get(filterName);

            FilterOperator filterOperator = null;
            if (valueNode.isArray() && valueNode.size() == 2 && valueNode.get(0).isTextual()) {
                // Возможно это случай, где valueNode имеет вид: ["=", ["1", "2"]] или ["=", "3"]
                filterOperator = FilterOperator.valueByLabel(valueNode.get(0).textValue());
            }

            if (filterOperator == null) {
                // valueNode имеет вид: ["1", "2"] или "3"
                filterOperator = FilterOperator.EQUALS;
            } else if (valueNode.get(1).isArray()) {
                // valueNode имеет вид: ["=", ["1", "2"]]
                valueNode = valueNode.get(1);
            } else {
                // valueNode имеет вид: ["=", "3"]
                ((ArrayNode) valueNode).remove(0);
            }

            filterNodes.add(new FilterNode(createValueFilter(filterName, filterOperator, valueNode)));

        }
        return new FilterNode(contextOperator, filterNodes);
    }

    private FilterOperator getVerifiedFilterOperator(String label) {
        FilterOperator filterOperator = FilterOperator.valueByLabel(label);
        if (filterOperator == null) {
            throw new CrnkFilterIncorrectException("Unexpected FilterOperator. Label = " + label);
        }
        return filterOperator;
    }

    private BinaryOperator getVerifiedBinaryOperator(String label) {
        BinaryOperator binaryOperator = BinaryOperator.valueByLabel(label);
        if (binaryOperator == null) {
            throw new CrnkFilterIncorrectException("Unexpected BinaryOperator. Label = " + label);
        }
        return binaryOperator;
    }

    private RawFilter createValueFilter(String name, FilterOperator filterOperator, JsonNode value) {
        Queue<String> path = new LinkedList<>(List.of(name.split("\\.")));
        if (path.size() > 1) {
            return new RawFilter(path.remove(), MATCH, matchFilterChain(path, filterOperator, value));
        } else {
        return new RawFilter(name, filterOperator, value);
        }
    }

    private JsonNode matchFilterChain(Queue<String> names, FilterOperator leafOperator, JsonNode leaf) {
        if (names.size() == 1) {
            return objectMapper.createArrayNode()
                    .add(names.remove())
                    .add(leafOperator.getLabel())
                    .add(leaf);
        }
        return objectMapper.createArrayNode()
                .add(names.remove())
                .add(MATCH.getLabel())
                .add(matchFilterChain(names, leafOperator, leaf));
    }
}
