package ru.yandex.partner.jsonapi.crnk.filter.parser;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;


import com.fasterxml.jackson.databind.ObjectMapper;
import io.crnk.core.queryspec.FilterSpec;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.jsonapi.crnk.CrnkModelFilters;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.IgnorableFilterParserException;

public class FilterNode {
    private static final Logger LOGGER = LoggerFactory.getLogger(FilterNode.class);

    private final BinaryOperator binaryOperator;
    private final List<FilterNode> filterNodes;
    private final RawFilter rawFilter;

    FilterNode(BinaryOperator binaryOperator, List<FilterNode> filterNodes) {
        this(binaryOperator, filterNodes, null);
    }

    FilterNode(RawFilter rawFilter) {
        this(null, null, rawFilter);
    }

    private FilterNode(BinaryOperator binaryOperator, List<FilterNode> filterNodes, RawFilter rawFilter) {
        this.binaryOperator = binaryOperator;
        this.rawFilter = rawFilter;
        this.filterNodes = filterNodes;
    }

    @NotNull
    public <M> CoreFilterNode<M> toCoreFilterNode(CrnkModelFilters<? super M> crnkModelFilters) {
        if (isValueNode()) {
            try {
                return new CoreFilterNode<>(rawFilter.toCoreFilter(crnkModelFilters));
            } catch (IgnorableFilterParserException e) {
                LOGGER.info("Replace CrnkModelFilterFilters with AlwaysFalseFilter");
                return CoreFilterNode.alwaysFalse();
            }
        } else {
            List<CoreFilterNode<? super M>> list = filterNodes.stream()
                    .map(filterNode -> filterNode.toCoreFilterNode(crnkModelFilters))
                    .collect(Collectors.toList());
            return new CoreFilterNode<>(binaryOperator, list);
        }
    }

    private boolean isValueNode() {
        return rawFilter != null;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof FilterNode)) {
            return false;
        }
        FilterNode that = (FilterNode) o;
        return binaryOperator == that.binaryOperator &&
                Objects.equals(filterNodes, that.filterNodes) &&
                Objects.equals(rawFilter, that.rawFilter);
    }

    @Override
    public int hashCode() {
        return Objects.hash(binaryOperator, filterNodes, rawFilter);
    }

    public static FilterNode fromFilterSpec(FilterSpec filterSpec) {
        //TODO
        FilterOperator filterOperator;
        if (filterSpec.getOperator() == io.crnk.core.queryspec.FilterOperator.EQ) {
            filterOperator = FilterOperator.EQUALS;
        } else {
            // TODO
            throw new RuntimeException("Unknown");
        }

        RawFilter rawFilter = new RawFilter(filterSpec.getPath().getElements().get(0), filterOperator,
                new ObjectMapper().valueToTree(filterSpec.getValue()));

        return new FilterNode(rawFilter);
    }
}
