package ru.yandex.partner.jsonapi.crnk.filter.parser.values;

import java.io.IOException;
import java.util.Collection;
import java.util.List;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.type.CollectionType;

import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterIncorrectException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.IgnorableFilterParserException;

public class DefaultCrnkFilterValueParser<T> implements CrnkFilterValueParser<T> {
    private final ObjectReader singleValueReader;
    private final ObjectReader collectionValueReader;

    public DefaultCrnkFilterValueParser(Class<T> tClass) {
        ObjectMapper objectMapper = new ObjectMapper();
        singleValueReader = objectMapper.readerFor(tClass);
        CollectionType collectionType = objectMapper.getTypeFactory().constructCollectionType(List.class, tClass);
        collectionValueReader = objectMapper.readerFor(collectionType);
    }

    @Override
    public Collection<T> toValues(JsonNode node) throws IgnorableFilterParserException {
        try {
            if (node.isNull()) {
                return List.of();
            }
            if (node.isObject()) {
                return handleUnexpectedNodeType(node);
            } else if (node.isArray()) {
                return readCollectionValue((ArrayNode) node);
            } else {
                return List.of(readSingleValue(node));
            }
        } catch (IOException e) {
            throw new CrnkFilterIncorrectException("Cannot parse values", e);
        }
    }

    /**
     * Чтение одиночного значения из JsonNode. Выделил в отдельный метод для возможности переопределения
     *
     * @param jsonNode объект содержащий одиночное значение
     * @return значение нужного типа
     * @throws IOException на случай ошибок чтения
     */
    protected T readSingleValue(JsonNode jsonNode) throws IOException {
        return singleValueReader.readValue(jsonNode);
    }

    /**
     * Чтение коллекции значений из JsonNode. Выделил в отдельный метод для возможности переопределения
     *
     * @param arrayNode объект содержащий коллекцию значений
     * @return коллекцию значений нужного типа
     * @throws IOException на случай ошибок чтения
     */
    protected List<T> readCollectionValue(ArrayNode arrayNode) throws IOException {
        return collectionValueReader.readValue(arrayNode);
    }

    protected Collection<T> handleUnexpectedNodeType(JsonNode node) {
        throw new CrnkFilterIncorrectException(
                "Unexpected node type " + node.getNodeType() + " : " + node);
    }
}
