package ru.yandex.partner.jsonapi.crnk.filter.parser.values;

import java.io.IOException;
import java.util.Collection;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;

import ru.yandex.direct.model.Model;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.StateFlag;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterIncorrectException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.IgnorableFilterParserException;
import ru.yandex.partner.libs.multistate.expression.ExpressionParsingException;
import ru.yandex.partner.libs.multistate.expression.MultistateExpressionParser;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

public class MultistateCrnkFilterValueParser<M extends Model, T extends StateFlag>
        implements CrnkFilterValueParser<Long> {

    private final MultistateGraph<M, T> multistateGraph;
    private final MultistateExpressionParser<T> multistateExpressionParser;
    private final ObjectReader stringValueReader;
    private final CrnkFilterValueParser<Long> numberValueParser;

    public MultistateCrnkFilterValueParser(MultistateGraph<M, T> multistateGraph,
                                           MultistateExpressionParser<T> multistateExpressionParser) {
        this.multistateGraph = multistateGraph;
        this.multistateExpressionParser = multistateExpressionParser;
        this.stringValueReader = new ObjectMapper().readerFor(String.class);
        this.numberValueParser = new DefaultCrnkFilterValueParser<>(Long.class);
    }

    @Override
    public Collection<Long> toValues(JsonNode node) throws IgnorableFilterParserException {
        try {
            if (node.isArray() && node.size() >= 1) {
                JsonNode tmpNode = node.get(0);

                if (tmpNode.isTextual() && node.size() > 1) {
                    throw new CrnkFilterIncorrectException("Cannot parse values: " + node.asText());
                } else if (tmpNode.isTextual()) {
                    node = tmpNode;
                }
            }

            if (node.isTextual()) {
                String expression = readStringValue(node);
                Predicate<Multistate<T>> predicate = expressionToPredicate(expression);
                return predicateToMultistateValues(predicate);
            } else {
                return numberValueParser.toValues(node).stream().distinct().toList();
            }
        } catch (IOException e) {
            throw new CrnkFilterIncorrectException("Cannot parse values: " + node.asText(), e);
        }
    }

    protected Predicate<Multistate<T>> expressionToPredicate(String expression) {
        try {
            return multistateExpressionParser.parseExpression(expression);
        } catch (ExpressionParsingException e) {
            throw new CrnkFilterIncorrectException(e.getMessage(), e);
        }
    }

    protected Set<Long> predicateToMultistateValues(Predicate<Multistate<T>> predicate) {
        return multistateGraph.getMultistatesForPredicate(predicate).stream()
                .map(Multistate::toMultistateValue).collect(Collectors.toSet());
    }

    protected String readStringValue(JsonNode jsonNode) throws IOException {
        return stringValueReader.readValue(jsonNode);
    }
}
