package ru.yandex.partner.jsonapi.crnk.filter.parser.values;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.type.CollectionType;

import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.CrnkFilterIncorrectException;
import ru.yandex.partner.jsonapi.crnk.filter.parser.exceptions.IgnorableFilterParserException;

public class NumberFilterValueParser implements CrnkFilterValueParser<Long> {
    private final ObjectReader singleValueReader;
    private final ObjectReader collectionValueReader;
    private final Pattern pattern;

    public NumberFilterValueParser() {
        pattern = Pattern.compile("\\d+");
        ObjectMapper objectMapper = new ObjectMapper();
        singleValueReader = objectMapper.readerFor(String.class);
        CollectionType collectionType = objectMapper.getTypeFactory().constructCollectionType(List.class, String.class);
        collectionValueReader = objectMapper.readerFor(collectionType);
    }

    public Collection<Long> toValues(JsonNode node) throws IgnorableFilterParserException {
        try {
            if (node.isNull()) {
                return List.of();
            }
            if (node.isObject()) {
                return handleUnexpectedNodeType(node);
            } else if (node.isArray()) {
                return readCollectionValue((ArrayNode) node);
            } else {
                return List.of(readSingleValue(node));
            }
        } catch (IOException e) {
            throw new CrnkFilterIncorrectException("Cannot parse values", e);
        }
    }

    protected Long readSingleValue(JsonNode jsonNode) throws IOException, IgnorableFilterParserException {
        String temp = singleValueReader.readValue(jsonNode);
        return longFromString(temp);
    }

    protected List<Long> readCollectionValue(ArrayNode arrayNode) throws IOException, IgnorableFilterParserException {
        List<String> list = collectionValueReader.readValue(arrayNode);
        return list.stream().map(this::longFromString).toList();
    }

    protected Collection<Long> handleUnexpectedNodeType(JsonNode node) {
        throw new CrnkFilterIncorrectException(
                "Unexpected node type " + node.getNodeType() + " : " + node);
    }


    protected Long longFromString(String str) throws IgnorableFilterParserException {
        Matcher matcher = pattern.matcher(str);
        if (matcher.lookingAt()) {
            try {
                return Long.parseLong(matcher.group());
            } catch (NumberFormatException e) {
                throw new IgnorableFilterParserException();
            }
        } else {
            throw new IgnorableFilterParserException();
        }
    }
}
