package ru.yandex.partner.jsonapi.crnk.kvstorefrontend.authorization;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.kvstorefrontend.model.KvStoreFrontend;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.authorization.actioncontext.CreateRequest;
import ru.yandex.partner.libs.authorization.actioncontext.RequestContext;
import ru.yandex.partner.libs.authorization.actioncontext.UpdateRequest;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.authorization.policy.base.Policy;

@ParametersAreNonnullByDefault
@Component
public class KvStoreFrontendAuthorizationPolicy implements Policy<KvStoreFrontend> {
    private final KvStoreOwnerViewRule ownerViewRule;
    private final KvStoreOwnerEditRule ownerEditRule;
    private final KvStoreViewAllRule viewAllRule;
    private final KvStoreEditAllRule editAllRule;

    @Autowired
    public KvStoreFrontendAuthorizationPolicy(KvStoreOwnerViewRule ownerViewRule,
                                              KvStoreOwnerEditRule ownerEditRule,
                                              KvStoreViewAllRule viewAllRule,
                                              KvStoreEditAllRule editAllRule) {
        this.ownerViewRule = ownerViewRule;
        this.ownerEditRule = ownerEditRule;
        this.viewAllRule = viewAllRule;
        this.editAllRule = editAllRule;
    }

    @Override
    public AuthorizationDecision<KvStoreFrontend> authorizeCreateRequest(RequestContext createRequest) {
        return authorizeUpdateRequest(createRequest);
    }

    @Override
    public AuthorizationDecision<KvStoreFrontend> authorizeUpdateRequest(RequestContext updateRequest) {
        return AuthorizationDecision.firstApplicable(
                editAllRule.authorize(updateRequest.getUserAuthentication()),
                ownerEditRule.authorize(updateRequest.getUserAuthentication())
        );
    }

    @Override
    public AuthorizationDecision<KvStoreFrontend> authorizeUpdateRequest(UpdateRequest<KvStoreFrontend> updateRequest) {
        UserAuthentication ua = updateRequest.getUserAuthentication();
        return AuthorizationDecision.firstApplicable(
                editAllRule.authorize(ua),
                AuthorizationDecision.denyOverrides(
                        ownerEditRule.authorize(ua, updateRequest.getCurrentInstance()),
                        ownerEditRule.authorize(ua, updateRequest.getUpdatePayload())
                )
        );
    }

    @Override
    public AuthorizationDecision<KvStoreFrontend> authorizeCreateRequest(CreateRequest<KvStoreFrontend> createRequest) {
        UserAuthentication ua = createRequest.getUserAuthentication();
        return AuthorizationDecision.firstApplicable(
                editAllRule.authorize(ua),
                ownerEditRule.authorize(ua, createRequest.getCreatePayload())
        );
    }

    @Override
    public AuthorizationDecision<KvStoreFrontend> authorizeGetRequest(RequestContext getRequest) {
        UserAuthentication userAuthentication = getRequest.getUserAuthentication();
        return AuthorizationDecision.firstApplicable(
                viewAllRule.authorize(userAuthentication),
                ownerViewRule.authorize(userAuthentication)
        );
    }
}
