package ru.yandex.partner.jsonapi.crnk.page;


import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.libs.auth.facade.UserAuthenticationProvider;
import ru.yandex.partner.libs.authorization.actioncontext.RequestContext;

import static java.util.function.Predicate.not;
import static one.util.streamex.MoreCollectors.onlyOne;
import static ru.yandex.partner.core.filter.CoreFilterNode.and;
import static ru.yandex.partner.core.filter.operator.FilterOperator.NOT_EQUALS;
import static ru.yandex.partner.jsonapi.crnk.page.authorization.PageAuthorizationPolicy.ASSISTANT_CAN_EDIT;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@ParametersAreNonnullByDefault
public abstract class BasePageReachabilityService implements PageReachabilityService {

    private final UserAuthenticationProvider authenticationProvider;

    public BasePageReachabilityService(
            UserAuthenticationProvider authenticationFacade
    ) {
        this.authenticationProvider = authenticationFacade;
    }

    @NotNull
    @Override
    public <P extends BasePage> CoreFilterNode<P> reachablePagesFilter(ApiModel<P> apiModel) {
        CoreFilterNode<P> filters = apiModel.getPolicy()
                .authorizeGetRequest(
                        new RequestContext(authenticationProvider.getUserAuthentication())
                                .withParam(ASSISTANT_CAN_EDIT, true)
                ).restriction();

        boolean shouldNotEditPartnerReadonlyField = hasField(apiModel, PageCrnkJsonFieldAliases.PARTNER_READ_ONLY) &&
                !authenticationProvider.getUserAuthentication().userHasRight(
                        PageRightsEnum.EDIT_FIELD__PARTNER_READ_ONLY.getFullRightName(apiModel.getResourceType())
                );

        if (shouldNotEditPartnerReadonlyField) {
            filters = and(filters,
                    CoreFilterNode.create(PageFilters.PARTNER_READ_ONLY, NOT_EQUALS, true)
            );
        }

        return filters;
    }

    public <P extends BasePage> CoreFilterNode<P> getNotProtectedFilter() {
        boolean isUserProtectedEditor = authenticationProvider.getUserAuthentication()
                .userHasRight(CoreConstants.EDIT_PROTECTED_PAGES_RIGHT_NAME);
        if (!isUserProtectedEditor) {
            return CoreFilterNode.create(
                    PageFilters.MULTISTATE,
                    FilterOperator.IN,
                    not(has(PageStateFlag.PROTECTED))
            );
        } else {
            return CoreFilterNode.neutral();
        }
    }

    private <P extends BasePage> boolean hasField(ApiModel<P> apiModel, String fieldName) {
        return apiModel.getFields().stream()
                .collect(onlyOne(f -> f.getJsonName().equals(fieldName)))
                .isPresent();
    }
}
