package ru.yandex.partner.jsonapi.crnk.page;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.domain.filter.DomainFilters;
import ru.yandex.partner.core.entity.domain.model.BaseDomain;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.entity.user.model.BaseUser;
import ru.yandex.partner.jsonapi.crnk.CrnkModelFilters;
import ru.yandex.partner.jsonapi.crnk.authorization.request.RequestAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.filter.provider.MobilePlatformDictionaryProvider;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilterName;
import ru.yandex.partner.jsonapi.crnk.filter.parser.CrnkFilterParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.MatchCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.user.UserCrnkModelFilters;
import ru.yandex.partner.jsonapi.crnk.user.authorization.UserAuthorizationPolicy;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.libs.authorization.actioncontext.RequestContext;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.authorization.policy.base.Policy;

@Component
public class MobileAppCrnkModelFilters implements CrnkModelFilters<MobileAppSettings> {

    private final Map<String, CrnkFilter<MobileAppSettings, ?>> filters = new HashMap<>();
    private final CrnkFilterParser crnkFilterParser;
    private final RequestAuthorizationService requestAuthorizationService;
    private final UserCrnkModelFilters userCrnkModelFilters;
    private final UserAuthorizationPolicy userAuthorizationPolicy;
    private final MobilePlatformDictionaryProvider mobilePlatformDictionaryProvider;

    @Autowired
    public MobileAppCrnkModelFilters(CrnkFilterParser crnkFilterParser,
                                     RequestAuthorizationService requestAuthorizationService,
                                     UserCrnkModelFilters userCrnkModelFilters,
                                     UserAuthorizationPolicy userAuthorizationPolicy,
                                     MobilePlatformDictionaryProvider mobilePlatformDictionaryProvider) {
        this.crnkFilterParser = crnkFilterParser;
        this.requestAuthorizationService = requestAuthorizationService;
        this.userCrnkModelFilters = userCrnkModelFilters;
        this.userAuthorizationPolicy = userAuthorizationPolicy;
        this.mobilePlatformDictionaryProvider = mobilePlatformDictionaryProvider;
        initFilters();
    }

    @Override
    public Map<String, CrnkFilter<MobileAppSettings, ?>> getFilters() {
        return filters;
    }

    private void initFilters() {
        // TODO: 25.05.2022 Сделать MatchCrnkFilterValueParser для мобильных приложений бином, когда
        // будем переносить API площадок и(или) приложений
        MatchCrnkFilterValueParser<BaseDomain> domainMatchCrnkFilterValueParser =
                new MatchCrnkFilterValueParser<>(
                        crnkFilterParser,
                        requestAuthorizationService,
                        () -> Map.of(
                                FilterNameEnum.DOMAIN.getName(),
                                CrnkFilter.builder(FilterNameEnum.DOMAIN)
                                        .toCrnkFilter(DomainFilters.DOMAIN),
                                FilterNameEnum.STORE_ID.getName(),
                                CrnkFilter.builder(FilterNameEnum.STORE_ID)
                                        .toCrnkFilter(DomainFilters.DOMAIN),
                                BlockFilterNameEnum.PLATFORM.getName(),
                                CrnkFilter.builder(BlockFilterNameEnum.PLATFORM)
                                        .toCrnkFilter(
                                                DomainFilters.TYPE,
                                                mobilePlatformDictionaryProvider)
                        ),
                        new Policy<>() {
                            @Override
                            public AuthorizationDecision<BaseDomain>
                            authorizeGetAllRequest(RequestContext getAllRequest) {
                                return AuthorizationDecision.permit();
                            }
                        }
                );

        MatchCrnkFilterValueParser<BaseUser> userMatchCrnkFilterValueParser =
                new MatchCrnkFilterValueParser<>(
                        crnkFilterParser,
                        requestAuthorizationService,
                        userCrnkModelFilters,
                        userAuthorizationPolicy
                );

        List<CrnkFilter<? super MobileAppSettings, ?>> filtersList = List.of(
                CrnkFilter.builder(FilterNameEnum.ID)
                        .toCrnkFilter(PageFilters.PAGE_ID),
                CrnkFilter.builder(FilterNameEnum.CAPTION)
                        .toCrnkFilter(PageFilters.CAPTION),
                CrnkFilter.builder(FilterNameEnum.ALL_DOMAIN)
                        .toCrnkFilter(PageFilters.ALL_DOMAIN),
                CrnkFilter.builder(FilterNameEnum.PAGE_ID)
                        .toCrnkFilter(PageFilters.PAGE_ID),
                CrnkFilter.builder(FilterNameEnum.DOMAIN)
                        .toCrnkFilterMatch(
                                PageFilters.DOMAIN,
                                domainMatchCrnkFilterValueParser,
                                CrnkFilter.builder(FilterNameEnum.DOMAIN)
                                        .setExposed(() -> true)
                        ),
                CrnkFilter.builder(FilterNameEnum.APPLICATION_ID)
                        .toCrnkFilter(PageFilters.DOMAIN_ID),
                CrnkFilter.builder(FilterNameEnum.APPLICATION)
                        .toCrnkFilterMatch(
                                PageFilters.DOMAIN,
                                domainMatchCrnkFilterValueParser,
                                CrnkFilter.builder(FilterNameEnum.STORE_ID),
                                CrnkFilter.builder(BlockFilterNameEnum.PLATFORM)
                        ),
                CrnkFilter.builder(FilterNameEnum.OWNER)
                        .toCrnkFilterMatch(
                                PageFilters.OWNER,
                                userMatchCrnkFilterValueParser,
                                CrnkFilter.builder(UserCrnkModelFilters.NameEnum.CLIENT_ID)
                        )
        );

        filtersList.forEach(filter -> filters.put(filter.getName(), (CrnkFilter<MobileAppSettings, ?>) filter));
    }

    public enum FilterNameEnum implements CrnkFilterName {
        CAPTION("caption"),
        ALL_DOMAIN("all_domain"),
        PAGE_ID("page_id"),
        DOMAIN("domain"),
        ID("id"),
        APPLICATION_ID("application_id"),
        STORE_ID("store_id"),
        APPLICATION("application"),
        OWNER("owner");

        private final String name;

        FilterNameEnum(String name) {
            this.name = name;
        }

        @Override
        public String getName() {
            return name;
        }
    }

}
