package ru.yandex.partner.jsonapi.crnk.page.authorization.actions;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.multistate.ContextPageMultistateGraph;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.AbstractActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionAuthEntry;
import ru.yandex.partner.jsonapi.crnk.page.PageComputableApiFieldsService;
import ru.yandex.partner.jsonapi.models.page.external.ApiContextPageMetaData;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.utils.StreamUtils;

@Component
@ParametersAreNonnullByDefault
public class ContextPageActionsAuthorizationService extends AbstractActionsAuthorizationService<ContextPage> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ContextPageActionsAuthorizationService.class);
    private static final String RIGHT_EDIT = "context_on_site_campaign_edit";

    private final PageComputableApiFieldsService pageComputableApiFieldsService;
    private final Map<String, ActionAuthEntry<ContextPage>> authEntryMap;

    @Autowired
    public ContextPageActionsAuthorizationService(ContextPageMultistateGraph graph,
                                                  ApiContextPageMetaData apiContextPageMetaData,
                                                  AuthenticationFacade authenticationFacade,
                                                  PageComputableApiFieldsService pageComputableApiFieldsService) {
        super(graph, apiContextPageMetaData, authenticationFacade, List.of());
        this.pageComputableApiFieldsService = pageComputableApiFieldsService;
        this.authEntryMap = initMap();
    }

    @Override
    // todo: вынести этот метод в базовый класс пейджового авторизатора, по аналогии с
    // BlockActionsAuthorizationService, когда будем делать внутренние площадки
    protected List<Boolean> authorizeWithEntry(UserAuthentication ua,
                                               ActionAuthEntry<ContextPage> authEntry,
                                               List<ContextPage> coreModels) {
        Optional<ContextPage> protectedPage = coreModels.stream()
                .filter(p -> p.getMultistate().hasFlag(PageStateFlag.PROTECTED))
                .findAny();

        if (!ua.userHasRight(RIGHT_EDIT)) {
            return Collections.nCopies(coreModels.size(), false);
        } else if (protectedPage.isPresent()) {
            LOGGER.info(
                    "User [ %s ] has tried to do action [ %s ] on PROTECTED page [ %s ]".formatted(
                            ua.getLogin(),
                            authEntry.getActionName(),
                            protectedPage.get().getId()
                    )
            );
            if (!ua.userHasRight(CoreConstants.EDIT_PROTECTED_PAGES_RIGHT_NAME)) {
                LOGGER.info(
                        "User [ %s ] has no rights to do action [ %s ] on PROTECTED page [ %s ]".formatted(
                                ua.getLogin(),
                                authEntry.getActionName(),
                                protectedPage.get().getId()
                        )
                );
                return Collections.nCopies(coreModels.size(), false);
            }
        }
        List<Boolean> notReadOnlyAssistant = coreModels.stream()
                .map(model -> !isCurUserReadOnlyAssistant(model))
                .collect(Collectors.toList());
        List<Boolean> basicCheck = super.authorizeWithEntry(ua, authEntry, coreModels);
        return StreamUtils.mergedStream(notReadOnlyAssistant, basicCheck, Boolean::logicalAnd)
                .collect(Collectors.toList());
    }

    public boolean isCurUserReadOnlyAssistant(ContextPage model) {
        if (model.getAssistants() == null) {
            throw new IllegalStateException("Assistants not loaded for page");
        }
        return pageComputableApiFieldsService.curUserIsReadAssistant(model.getAssistants());
    }

    @Override
    protected Map<String, ActionAuthEntry<ContextPage>> getAuthEntryMap() {
        return authEntryMap;
    }

    private Map<String, ActionAuthEntry<ContextPage>> initMap() {
        // TODO: this is not complete list of actions
        return ActionAuthEntry.buildMap("context_on_site", List.of(
                ActionAuthEntry.<ContextPage>builder()
                        // action authorization used in RtbBlock
                        .setActionName(PageActionsEnum.RESTORE_BLOCK)
                        .setExposed(false)
                        .setRightRequired(false)
        ));
    }
}
