package ru.yandex.partner.jsonapi.crnk.paging;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;

import io.crnk.core.exception.BadRequestException;
import io.crnk.core.exception.ParametersDeserializationException;
import io.crnk.core.queryspec.pagingspec.NumberSizePagingBehavior;
import io.crnk.core.queryspec.pagingspec.NumberSizePagingSpec;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.stereotype.Component;

import ru.yandex.partner.jsonapi.messages.ControllerMsg;
import ru.yandex.partner.jsonapi.messages.MethodMsg;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

/**
 * NumberSizePagingBehavior, который переводит сообщение об ошибке, если page[size] больше чем позволено
 * Также проверяет, что size и number не меньше 1
 */
@Component
public class RestrictedNumberSizePagingBehaviour extends NumberSizePagingBehavior implements MessageSourceAware {

    protected MessageSourceAccessor messages = new MessageSourceAccessor(new DelegatingMessageSource());

    @Override
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        messages = new MessageSourceAccessor(messageSource);
    }

    @Override
    public NumberSizePagingSpec deserialize(Map<String, Set<String>> parameters) {
        NumberSizePagingSpec pagingSpec = deserializeInternal(parameters);
        validate(pagingSpec);
        return pagingSpec;
    }

    private NumberSizePagingSpec deserializeInternal(Map<String, Set<String>> parameters) {
        try {
            return super.deserialize(parameters);
        } catch (BadRequestException e) {
            if (e instanceof ParametersDeserializationException) {
                throw e;
            }
            throw new BadRequestException(
                    messages.getMessage(MsgWithArgs.of(ControllerMsg.PARAMETER_PAGE_SIZE_EXCEEDED, getMaxPageLimit())));
        }
    }

    private void validate(NumberSizePagingSpec pagingSpec) {
        if (pagingSpec.getSize() < 1 || pagingSpec.getNumber() < 1) {
            Optional<String> paramName = super.serialize(pagingSpec, null).entrySet().stream().filter(
                    e -> e.getValue().stream().filter(Objects::nonNull).map(Long::valueOf).anyMatch(l -> l < 1)
            ).map(Map.Entry::getKey).findFirst();
            throw new BadRequestException(paramName.map(param ->
                    messages.getMessage(MsgWithArgs.of(MethodMsg.PARAMETER_MUST_BE_WHOLE_POSITIVE_NUMBER, param)
                    )).orElse(null));
        }
    }

}
