package ru.yandex.partner.jsonapi.crnk.user;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.mapstruct.Mapper;
import org.mapstruct.Mapping;
import org.mapstruct.Named;
import org.mapstruct.ReportingPolicy;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;
import org.springframework.context.support.DelegatingMessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.jsonapi.crnk.user.models.RoleCrnk;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.rbac.role.RoleSet;

@Mapper(componentModel = "spring", unmappedTargetPolicy = ReportingPolicy.IGNORE)
@Component
public abstract class RoleCrnkMapper {

    private MessageSourceAccessor messages = new MessageSourceAccessor(new DelegatingMessageSource());

    @Autowired
    public void setMessageSource(@Nonnull MessageSource messageSource) {
        messages = new MessageSourceAccessor(messageSource);
    }

    @Mapping(source = "nameMsg", target = "name")
    @Mapping(source = "conflictRoleForSet", target = "conflictRoleForSet", qualifiedByName = "mapRolesToRoleId")
    @Mapping(source = "requiredRoleForSet", target = "requiredRoleForSet", qualifiedByName = "mapRolesToRoleId")
    public abstract RoleCrnk roleToRoleCrnk(Role role);

    public Role roleCrnkToRole(RoleCrnk role) {
        Optional<Role> roleById = RoleSet.getRoleById(role.getRoleId());
        return roleById.orElse(null);
    };

    protected String map(GettextMsg value) {
        return messages.getMessage(value);
    }

    @Named("mapRolesToRoleId")
    protected List<Long> rolesToRoleId(Set<Role> roles) {
        return roles == null ? null : roles.stream().map(Role::getRoleId).collect(Collectors.toList());
    }

    @Named("mapRoleIdToRoles")
    protected Set<Role> roleIdToRoles(List<Long> roleIds) {
        return roleIds == null ? null : roleIds.stream()
                .map(RoleSet::getRoleById)
                .map(o -> o.orElse(null))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }
}
