package ru.yandex.partner.jsonapi.crnk.user.authorization;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.rbac.right.RightService;
import ru.yandex.partner.libs.rbac.role.RoleSet;

@Component
public class ManagerRoleByRightRule {
    private final RightService rightService;

    private static final Map<String, Role> RIGHT_ROLE_MAP = Map.ofEntries(
            Map.entry("users_view_all_developers", RoleSet.DEVELOPER),
            Map.entry("users_view_all_an_internal_managers", RoleSet.INTERNAL_YAN_MANAGER),
            Map.entry("users_view_all_dsp_partners", RoleSet.DSP_PARTNER),
            Map.entry("users_view_all_dsp_managers", RoleSet.DSP_MANAGER),
            Map.entry("users_view_all_an_partners", RoleSet.SITE_PARTNER),
            Map.entry("users_view_all_video_an_partners", RoleSet.VIDEO_PARTNER),
            Map.entry("users_view_all_mobile_an_partners", RoleSet.MOBILE_PARTNER),
            Map.entry("users_view_all_adblock_an_partners", RoleSet.ADBLOCK_PARTNER),
            Map.entry("users_view_all_tutby_aggregators", RoleSet.TUTBY),
            Map.entry("users_view_all_indoor_an_partners", RoleSet.INDOOR_PARTNER),
            Map.entry("users_view_all_outdoor_an_partners", RoleSet.OUTDOOR_PARTNER)
    );

    public ManagerRoleByRightRule(RightService rightService) {
        this.rightService = rightService;
    }

    public AuthorizationDecision<User> authorize(UserAuthentication ua) {
        HashSet<Long> availableIds = RIGHT_ROLE_MAP
                .entrySet()
                .stream()
                .filter(e -> ua.userHasRight(e.getKey()))
                .map(e -> e.getValue().getRoleId())
                .collect(Collectors.toCollection(HashSet::new));

        if (ua.userHasRight("yndx_login")) {
            Set<Long> yndxLoginRoleIds = rightService
                    .getRightRoles("yndx_login")
                    .stream()
                    .map(Role::getRoleId)
                    .collect(Collectors.toSet());

            availableIds.addAll(yndxLoginRoleIds);
        }

        if (!availableIds.isEmpty()) {
            return AuthorizationDecision.restrictedPermit(CoreFilterNode.create(
                    UserFilters.ROLE_ID,
                    FilterOperator.IN,
                    availableIds
            ));
        } else {
            return AuthorizationDecision.unapplicable();
        }
    }
}
