package ru.yandex.partner.jsonapi.crnk.user.authorization;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.libs.authorization.actioncontext.CreateResult;
import ru.yandex.partner.libs.authorization.actioncontext.GetAllResult;
import ru.yandex.partner.libs.authorization.actioncontext.GetResult;
import ru.yandex.partner.libs.authorization.actioncontext.RequestContext;
import ru.yandex.partner.libs.authorization.actioncontext.UpdateResult;
import ru.yandex.partner.libs.authorization.decision.AuthorizationDecision;
import ru.yandex.partner.libs.authorization.policy.base.Policy;

@Component
public class UserAuthorizationPolicy implements Policy<User> {

    private final EditAllRule<User> editAllRule;
    private final ActionAddRule<User> actionAddRule;
    private final ViewAllRule<User> viewAllRule;
    private final TutbyRule tutbyRule;
    private final DenyRule<User> denyRule;
    private final SelfAccessRule selfAccessRule;
    private final AdinsideRule adinsideRule;
    private final OwnersForAssistantRule ownersForAssistantRule;
    private final ManagerRoleByRightRule managerRoleByRightRule;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public UserAuthorizationPolicy(EditAllRule<User> editAllRule, ActionAddRule<User> actionAddRule,
                                   ViewAllRule<User> viewAllRule, TutbyRule tutbyRule, DenyRule<User> denyRule,
                                   SelfAccessRule selfAccessRule, AdinsideRule adinsideRule,
                                   OwnersForAssistantRule ownersForAssistantRule,
                                   ManagerRoleByRightRule managerRoleByRightRule) {
        this.editAllRule = editAllRule;
        this.actionAddRule = actionAddRule;
        this.viewAllRule = viewAllRule;
        this.tutbyRule = tutbyRule;
        this.denyRule = denyRule;
        this.selfAccessRule = selfAccessRule;
        this.adinsideRule = adinsideRule;
        this.ownersForAssistantRule = ownersForAssistantRule;
        this.managerRoleByRightRule = managerRoleByRightRule;
    }


    @Override
    public AuthorizationDecision<User> authorizeCreateRequest(RequestContext createRequest) {
        return AuthorizationDecision.firstApplicable(
                actionAddRule.authorize(createRequest.getUserAuthentication()),
                denyRule.authorize()
        );
    }

    @Override
    public AuthorizationDecision<User> authorizeCreateResult(CreateResult<User> createResult) {
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<User> authorizeGetRequest(RequestContext getRequest) {
        return AuthorizationDecision.firstApplicable(
                viewAllRule.authorize(getRequest.getUserAuthentication()),
                AuthorizationDecision.permitOverrides(
                        managerRoleByRightRule.authorize(getRequest.getUserAuthentication()),
                        adinsideRule.authorize(getRequest.getUserAuthentication()),
                        tutbyRule.authorize(getRequest.getUserAuthentication()),
                        ownersForAssistantRule.authorize(getRequest.getUserAuthentication()),
                        selfAccessRule.authorize(getRequest.getUserAuthentication())
                ),
                denyRule.authorize()
        );
    }

    @Override
    public AuthorizationDecision<User> authorizeGetResult(GetResult<User> getResult) {
        // TODO RtbBlock authorizeGetResult
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<User> authorizeGetAllResult(GetAllResult<User> getAllResult) {
        // TODO RtbBlock authorizeGetResult
        return AuthorizationDecision.permit();
    }

    @Override
    public AuthorizationDecision<User> authorizeUpdateRequest(RequestContext updateRequest) {
        return AuthorizationDecision.firstApplicable(
                editAllRule.authorize(updateRequest.getUserAuthentication()),
                selfAccessRule.authorize(updateRequest.getUserAuthentication())
        );
    }

    @Override
    public AuthorizationDecision<User> authorizeUpdateResult(UpdateResult<User> updateResult) {
        return AuthorizationDecision.permit();
    }
}
