package ru.yandex.partner.jsonapi.filter.csrf;

import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.partner.jsonapi.messages.RestApiMsg;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.AuthenticationMethod;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;

@Component
@ParametersAreNonnullByDefault
public class StatelessCsrfFilter extends OncePerRequestFilter {

    private static final String X_CSRF_TOKEN = "X-Frontend-Authorization";
    private final AuthenticationFacade authenticationFacade;

    public StatelessCsrfFilter(AuthenticationFacade authenticationFacade) {
        this.authenticationFacade = authenticationFacade;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {

        UserAuthentication userAuthentication = authenticationFacade.getUserAuthentication();
        if (userAuthentication != null &&
                userAuthentication.getAuthenticationMethod() == AuthenticationMethod.AUTH_VIA_COOKIES) {

            String serverCsrfToken = generateServerToken();
            response.setHeader(X_CSRF_TOKEN, serverCsrfToken);

            if (!"GET".equals(request.getMethod())) {
                final String csrfTokenValue = request.getHeader(X_CSRF_TOKEN);
                String csrfToken = csrfTokenValue != null ?
                        csrfTokenValue.replaceFirst("token ", "") : "";

                if (!serverCsrfToken.equals(csrfToken)) {
                    throw new I18nResponseStatusException(HttpErrorStatusEnum.ERROR__NEED_AUTH,
                            RestApiMsg.INCORRECT_CSRF_TOKEN);
                }
            }
        }

        filterChain.doFilter(request, response);
    }

    private String generateServerToken() {
        long userId = authenticationFacade.getUserAuthentication().getUid();
        // Логика перенесена из perl - получаем timestamp в секундах,
        // округленный до 5 минут в нижнюю сторону
        long seconds = System.currentTimeMillis() / 1000;
        long time = (seconds / 300) * 300;
        return DigestUtils.md5Hex(String.format("%d_%d", time, userId));
    }
}
