package ru.yandex.partner.jsonapi.filter.limit;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nonnull;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.springframework.web.filter.OncePerRequestFilter;

import ru.yandex.partner.jsonapi.messages.RestApiMsg;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.exceptions.I18nResponseStatusException;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

public class RequestLimitFilter extends OncePerRequestFilter {
    private static final Integer MAX_REQUESTS_AT_SAME_TIME = 6;
    private final AuthenticationFacade authenticationFacade;
    private final Map<String, Integer> userLimitMap;
    private final Integer defaultLimit;
    private final LoadingCache<Long, AtomicInteger> perMinuteLimitCache;
    private final LoadingCache<Long, AtomicInteger> atSameTimeLimitCache;

    /**
     * @param defaultLimit лимит по умолчанию или отрицательное число, если нет ограничений
     */
    public RequestLimitFilter(
            AuthenticationFacade authenticationFacade,
            int defaultLimit,
            Map<String, Integer> userLimitMap
    ) {
        this.authenticationFacade = authenticationFacade;
        this.defaultLimit = defaultLimit < 0 ? Integer.MAX_VALUE : defaultLimit;
        this.userLimitMap = userLimitMap;

        this.perMinuteLimitCache = Caffeine.newBuilder()
                .expireAfterWrite(1L, TimeUnit.MINUTES)
                .build(uid -> new AtomicInteger(0));

        this.atSameTimeLimitCache = Caffeine.newBuilder()
                .expireAfterAccess(5L, TimeUnit.MINUTES)
                .build(uid -> new AtomicInteger(0));

    }

    @Override
    protected void doFilterInternal(@Nonnull HttpServletRequest request,
                                    @Nonnull HttpServletResponse response,
                                    @Nonnull FilterChain filterChain)
            throws ServletException, IOException {
        UserAuthentication userAuthentication = authenticationFacade.getUserAuthentication();
        if (userAuthentication == null || userAuthentication.isSelfAuth()) {
            filterChain.doFilter(request, response);
            return;
        }
        String login = userAuthentication.getLogin();
        long uid = userAuthentication.getUid();
        int limit = getUserLimit(login);

        int perMinuteCount = perMinuteLimitCache.get(uid).incrementAndGet();
        if (perMinuteCount > limit) {
            throw new I18nResponseStatusException(HttpErrorStatusEnum.ERROR__MANY_REQ,
                    MsgWithArgs.of(RestApiMsg.ALLOWED_REQUEST_PER_MINUTE, limit)
            );
        }

        AtomicInteger atSameTimeCounter = atSameTimeLimitCache.get(uid);
        if (atSameTimeCounter.incrementAndGet() > MAX_REQUESTS_AT_SAME_TIME) {
            atSameTimeCounter.decrementAndGet();
            throw new I18nResponseStatusException(HttpErrorStatusEnum.ERROR__MANY_REQ);
        }
        try {
            filterChain.doFilter(request, response);
        } finally {
            atSameTimeCounter.decrementAndGet();
        }
    }

    private int getUserLimit(String login) {
        return userLimitMap.getOrDefault(login, defaultLimit);
    }
}
