package ru.yandex.partner.jsonapi.jackson;

import java.io.IOException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.deser.DeserializationProblemHandler;
import com.fasterxml.jackson.databind.deser.ValueInstantiator;
import com.fasterxml.jackson.databind.exc.UnrecognizedPropertyException;

public class CollectingErrorsDeserializationProblemHandler extends DeserializationProblemHandler {
    private List<DeserializationExceptionData> jsonMappingExceptions = new LinkedList<>();

    /**
     * Сюда попадаем, когда пришел неизвестный тег
     * <p>
     * Полезные ссылки
     * {@link DeserializationContext#handleUnknownProperty(JsonParser, JsonDeserializer, Object, String)}
     */
    @Override
    public boolean handleUnknownProperty(DeserializationContext ctxt, JsonParser p, JsonDeserializer<?> deserializer,
                                         Object beanOrClass, String propertyName) throws IOException {
        // Do we know properties that are expected instead?
        Collection<Object> propIds = (deserializer == null) ? null : deserializer.getKnownPropertyNames();
        var exception = UnrecognizedPropertyException.from(p, beanOrClass, propertyName, propIds);
        jsonMappingExceptions.add(
                new DeserializationExceptionData(
                        ctxt.getParser().getParsingContext(),
                        null,
                        null
                )
        );
        skipTokens(p);
        return true;
    }


    /**
     * Сюда попадаем, когда ожидалась строка определенного формата (число, дата и тп), но пришел не корректный формат
     */
    @Override
    public Object handleWeirdStringValue(DeserializationContext ctxt, Class<?> targetType, String valueToConvert,
                                         String failureMsg) {
        jsonMappingExceptions.add(
                new DeserializationExceptionData(
                        ctxt.getParser().getParsingContext(),
                        targetType,
                        valueToConvert
                )
        );
        return null;
    }

    /**
     * Сюда попадаем, когда число не можем перевести в другой тип (дату и тп)
     */
    @Override
    public Object handleWeirdNumberValue(DeserializationContext ctxt, Class<?> targetType, Number valueToConvert,
                                         String failureMsg) {
        jsonMappingExceptions.add(
                new DeserializationExceptionData(
                        ctxt.getParser().getParsingContext(),
                        targetType,
                        valueToConvert
                )
        );
        return null;
    }


    /**
     * Сюда попадаем когда ожидался простой тип (число, строка и тп), а прислали сложный (объект, массив и тп)
     * <p>
     * Полезные ссылки
     * {@link DeserializationContext#handleUnexpectedToken(Class, JsonToken, JsonParser, String, Object...)}
     * {@link DeserializationContext#reportInputMismatch(BeanProperty, String, Object...)}
     */
    @Override
    public Object handleUnexpectedToken(DeserializationContext ctxt, JavaType targetType, JsonToken t, JsonParser p,
                                        String failureMsg) throws IOException {
        jsonMappingExceptions.add(
                new DeserializationExceptionData(
                        ctxt.getParser().getParsingContext(),
                        targetType.getRawClass(),
                        null
                )
        );

        skipTokens(p);
        return null;
    }


    /**
     * Сюда попадаем, когда ожидался сложный тип (объект, массив и тп), а прислали простой (число, строка и тп)
     */
    @Override
    public Object handleMissingInstantiator(DeserializationContext ctxt, Class<?> instClass,
                                            ValueInstantiator valueInsta, JsonParser p, String msg) {
        jsonMappingExceptions.add(
                new DeserializationExceptionData(
                        ctxt.getParser().getParsingContext(),
                        instClass,
                        null
                )
        );
        return null;
    }

    public List<DeserializationExceptionData> getAndClearCollectedErrors() {
        var collectedErrors = jsonMappingExceptions;
        jsonMappingExceptions = new LinkedList<>();
        return collectedErrors;
    }

    public void addDeserializationError(DeserializationExceptionData error) {
        jsonMappingExceptions.add(error);
    }

    private void skipTokens(JsonParser jsonParser) throws IOException {
        JsonToken jsonToken = jsonParser.getCurrentToken();
        if (JsonToken.START_ARRAY.equals(jsonToken)) {
            skipTokens(jsonParser, JsonToken.START_ARRAY, JsonToken.END_ARRAY);
        }
        if (JsonToken.START_OBJECT.equals(jsonToken)) {
            skipTokens(jsonParser, JsonToken.START_OBJECT, JsonToken.END_OBJECT);
        }
    }

    /**
     * Пропускает токены, которые уже не нужно читать
     */
    private void skipTokens(JsonParser jsonParser, JsonToken startToken, JsonToken endToken) throws IOException {
        // это условие выполнено jsonParser.getCurrentToken() == startToken
        // поэтому count = 1
        int count = 1;
        JsonToken curToken;
        do {
            curToken = jsonParser.nextToken();
            if (startToken.equals(curToken)) {
                count++;
            } else if (endToken.equals(curToken)) {
                count--;
            }
        } while (count != 0 && curToken != null);
    }
}
