package ru.yandex.partner.jsonapi.jackson;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.core.validation.defects.presentation.CommonValidationMsg;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;
import ru.yandex.partner.jsonapi.messages.CommonMsg;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.utils.EnumTypeFromLiteralDeserializer;

import static com.fasterxml.jackson.databind.DeserializationFeature.ACCEPT_FLOAT_AS_INT;

public class CollectingErrorsParser {
    private static final Set<Class<?>> NUMBER_CLASSES = Set.of(
            Byte.class,
            Short.class,
            Integer.class,
            Long.class,
            Float.class,
            Double.class,
            BigDecimal.class
    );

    private final CollectingErrorsDeserializationProblemHandler collectingErrorsDeserializationProblemHandler;
    private final ObjectMapper objectMapper;
    private final boolean dropUnknownFieldInPath;
    private final Multimap<Model, ModelProperty<?, ?>> parsedPropertiesPerModel
            = Multimaps.newSetMultimap(new IdentityHashMap<>(), HashSet::new);

    public CollectingErrorsParser() {
        this(new Builder());
    }

    public CollectingErrorsParser(Builder builder) {
        this.dropUnknownFieldInPath = builder.dropUnknownFieldInPath;
        this.collectingErrorsDeserializationProblemHandler = new CollectingErrorsDeserializationProblemHandler();
        this.objectMapper = ObjectMapperFactory
                .createObjectMapper()
                .registerModule(new SimpleModule()
                        .addDeserializer(
                                DesignTemplatesType.class,
                                new EnumTypeFromLiteralDeserializer<>(DesignTemplatesType.class))
                        .setDeserializerModifier(new ModelPropertyListeningBeanDeserializerModifier(
                                parsedPropertiesPerModel
                        ))
                )
                .addHandler(collectingErrorsDeserializationProblemHandler)
                .disable(ACCEPT_FLOAT_AS_INT);
    }

    public Object parse(JsonNode jsonNode, JavaType javaType) throws IOException {
        if (Objects.isNull(jsonNode)) {
            return null;
        }
        JsonParser jsonParser = objectMapper.treeAsTokens(jsonNode);
        try {
            return objectMapper.readValue(jsonParser, javaType);
        } catch (InvalidFormatException ife) {
            collectingErrorsDeserializationProblemHandler.addDeserializationError(
                    new DeserializationExceptionData(
                            ((JsonParser) ife.getProcessor()).getParsingContext(),
                            ife.getTargetType(),
                            ife.getValue()
                    )
            );
        } catch (IOException e) {
            collectingErrorsDeserializationProblemHandler.addDeserializationError(
                    new DeserializationExceptionData(
                            jsonParser.getParsingContext(),
                            javaType.getRawClass(),
                            jsonNode.toString())
            );
        }
        return null;
    }

    public List<DefectInfo<Defect>> getAndClearCollectedDefects(String jsonName) {
        return collectingErrorsDeserializationProblemHandler.getAndClearCollectedErrors()
                .stream()
                .map(exceptionData -> {
                    Path path = toPath(jsonName, exceptionData.getPath());

                    // для обратной совместимости валидации отбрасываем сам некорректный ключ
                    Path fixedPath = path;
                    if (dropUnknownFieldInPath
                            && exceptionData.getTargetClass() == null
                            && path.getNodes().size() > 0) {
                        fixedPath = new Path(path.getNodes().subList(0, path.getNodes().size() - 1));
                    }

                    return DefectInfoBuilder.of(parseDefectMessage(exceptionData.getTargetClass(),
                                    exceptionData.getValueToConvert(),
                                    path
                            ))
                            .withPath(fixedPath)
                            .<Defect>build();
                })
                .collect(Collectors.toList());
    }

    private Path toPath(String roodNode, String[] path) {
        var list = new ArrayList<PathNode.Field>(path.length + 1);

        list.add(new PathNode.Field(roodNode));
        for (String reference : path) {
            list.add(new PathNode.Field(reference));
        }

        return new Path(list);
    }

    private GettextMsg parseDefectMessage(Class<?> expectedClass, Object value, Path path) {
        if (expectedClass == null) {
            return MsgWithArgs.of(CommonValidationMsg.UNKNOWN_KEY, path.getFieldName());
        } else if (expectedClass.isPrimitive()) {
            if (boolean.class.isAssignableFrom(expectedClass)) {
                return CommonMsg.DATA_MUST_BE_BOOLEAN;
            } else if (char.class.isAssignableFrom(expectedClass)) {
                return CommonMsg.DATA_MUST_BE_STRING;
            } else {
                // number
                return CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER;
            }
        } else {
            if (Boolean.class.isAssignableFrom(expectedClass)) {
                return CommonMsg.DATA_MUST_BE_BOOLEAN;
            } else if (String.class.isAssignableFrom(expectedClass)
                    || Character.class.isAssignableFrom(expectedClass)) {
                return CommonMsg.DATA_MUST_BE_STRING;
            } else if (Collection.class.isAssignableFrom(expectedClass) || expectedClass.isArray()) {
                return CommonMsg.DATA_MUST_BE_ARRAY;
            } else if (NUMBER_CLASSES.contains(expectedClass)) {
                return CommonValidationMsg.DATA_MUST_BE_INTEGER_NUMBER;
            } else {
                return CommonMsg.DATA_MUST_BE_HASH;
            }
        }
    }

    public Multimap<Model, ModelProperty<?, ?>> getParsedPropertiesPerModel() {
        return parsedPropertiesPerModel;
    }

    public static class Builder {
        private boolean dropUnknownFieldInPath = true;

        public Builder setDropUnknownFieldInPath(boolean dropUnknownFieldInPath) {
            this.dropUnknownFieldInPath = dropUnknownFieldInPath;
            return this;
        }
    }
}
