package ru.yandex.partner.jsonapi.jackson;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.lang.reflect.Member;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.DeserializationConfig;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.BeanDeserializerBuilder;
import com.fasterxml.jackson.databind.deser.BeanDeserializerModifier;
import com.fasterxml.jackson.databind.deser.SettableBeanProperty;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.utils.ModelPropertyUtils;

/**
 * Модификатор десериализаторов бинов-наследников класса Model, сохраняющий информацию о свойствах
 * {@link ModelProperty}, которые были установлены в процессе парсинга.
 */
class ModelPropertyListeningBeanDeserializerModifier extends BeanDeserializerModifier {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModelPropertyListeningBeanDeserializerModifier.class);

    private final Multimap<Model, ModelProperty<?, ?>> parsedPropertiesPerModel;

    ModelPropertyListeningBeanDeserializerModifier(
            Multimap<Model, ModelProperty<?, ?>> parsedPropertiesPerModel) {
        this.parsedPropertiesPerModel = parsedPropertiesPerModel;
    }

    @Override
    public BeanDeserializerBuilder updateBuilder(
            DeserializationConfig config,
            BeanDescription beanDesc,
            BeanDeserializerBuilder builder) {
        if (!Model.class.isAssignableFrom(beanDesc.getBeanClass())) {
            return builder;
        }

        var modelClass = (Class<? extends Model>) beanDesc.getBeanClass();

        var existingProps = Lists.newArrayList(builder.getProperties());
        BeanInfo beanInfo;
        try {
            beanInfo = Introspector.getBeanInfo(modelClass);
        } catch (IntrospectionException e) {
            throw new RuntimeException("Could not introspect Model " + modelClass, e);
        }

        Map<Member, PropertyDescriptor> propsBySetters =
                Stream.of(beanInfo.getPropertyDescriptors())
                        .filter(it -> it.getWriteMethod() != null)
                        .collect(Collectors.toMap(
                                PropertyDescriptor::getWriteMethod,
                                Function.identity()
                        ));

        Map<String, ModelProperty<?, ?>> modelPropertiesByName = ModelPropertyUtils.allModelProperties(modelClass)
                .stream().collect(Collectors.toMap(
                        ModelProperty::name,
                        Function.identity()
                ));

        for (SettableBeanProperty existingProp : existingProps) {
            var iProperty = propsBySetters.get(
                    existingProp.getMember().getMember()
            );

            if (iProperty == null) {
                LOGGER.warn("Property {} for model {} may be missing a setter and won't be listened for changes",
                        existingProp.getName(),
                        modelClass
                );
                continue;
            }

            builder.addOrReplaceProperty(new ListenableProperty(
                            existingProp,
                            modelClass,
                            (ModelProperty<? extends Model, Object>) modelPropertiesByName.get(iProperty.getName()),
                            parsedPropertiesPerModel::put
                    ),
                    false
            );
        }
        return builder;
    }

    private interface PropertyListener {
        void beforeSet(Model instance, ModelProperty<? extends Model, Object> modelProperty);
    }

    private static class ListenableProperty extends SettableBeanProperty.Delegating {
        private final SettableBeanProperty existingProp;
        private final Class<? extends Model> modelClass;
        private final ModelProperty<? extends Model, Object> modelProperty;
        private final PropertyListener propertyListener;

        ListenableProperty(SettableBeanProperty existingProp,
                           Class<? extends Model> modelClass,
                           ModelProperty<? extends Model, Object> modelProperty,
                           PropertyListener propertyListener) {
            super(existingProp);
            this.existingProp = existingProp;
            this.modelClass = modelClass;
            this.modelProperty = modelProperty;
            this.propertyListener = propertyListener;
        }

        @Override
        protected SettableBeanProperty withDelegate(SettableBeanProperty d) {
            return new ListenableProperty(d, modelClass, modelProperty, propertyListener);
        }

        @Override
        public void set(Object instance, Object value) throws IOException {
            beforeSet((Model) instance);
            existingProp.set(instance, value);
        }

        @Override
        public Object setAndReturn(Object instance, Object value) throws IOException {
            beforeSet((Model) instance);
            return existingProp.setAndReturn(instance, value);
        }

        @Override
        public Object deserializeSetAndReturn(JsonParser p,
                                              DeserializationContext ctxt,
                                              Object instance) throws IOException {
            beforeSet((Model) instance);
            return existingProp.deserializeSetAndReturn(p, ctxt, instance);
        }

        @Override
        public void deserializeAndSet(JsonParser p, DeserializationContext ctxt,
                                      Object instance) throws IOException {
            beforeSet((Model) instance);
            existingProp.deserializeAndSet(p, ctxt, instance);
        }

        private void beforeSet(Model instance) {
            propertyListener.beforeSet(instance, modelProperty);
        }
    }
}
