package ru.yandex.partner.jsonapi.messages;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;

import org.springframework.http.HttpStatus;

import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.i18n.GettextMsgPayload;

import static ru.yandex.partner.libs.i18n.GettextMsgPayload.Builder.msg;

public enum JsonapiErrorMsg implements GettextMsg {
    BAD_REQUEST(msg("Bad Request")),
    NOT_ACCEPTABLE(msg("Not Acceptable")),
    NOT_IMPLEMENTED(msg("Not implemented")),
    TOO_MANY_REQUESTS(msg("Too Many Requests")),
    UNAUTHORIZED(msg("Unauthorized")),
    UNSUPPORTED_MEDIA_TYPE(msg("Unsupported Media Type")),
    INTERNAL_SERVER_ERROR(msg("Internal Server Error")),
    NOT_FOUND(msg("Not found"), "lib__WebInterface__Controller__Settings.pm"),
    CONFLICT(msg("Conflict"), "lib__RestApi__Controller.pm"),
    VALIDATION_ERROR(msg("Validation error"), "lib__RestApi__Controller.pm"),
    CANNOT_EDIT_FIELDS(
            msg("You can not edit the following fields: %s"),
            "lib__Application__Model__Product__VideoAN__Site.pm"
    ),
    CANNOT_DO_ACTION(
            msg("Cannot do action \"%s\"."),
            "lib__QBit__Application__Model__Multistate__DB.pm"
    ),
    DUPLICATED_FIELDS(msg("Following fields duplicated: %s"), "lib__RestApi__Controller.pm");

    private static final String KEYSET_NAME = "lib__RestApi__Errors.pm";

    private static final Map<HttpStatus, JsonapiErrorMsg> STATUS_MESSAGE_MAP = new EnumMap<>(HttpStatus.class);
    private static final Map<Integer, JsonapiErrorMsg> ERROR_ID_MESSAGE_MAP = new HashMap<>();

    static {
        fillStatusMessageMap();
        fillErrorIdMessageMap();
    }

    private final GettextMsgPayload payload;
    private final String keyset;

    JsonapiErrorMsg(GettextMsgPayload.Builder builder) {
        this(builder, KEYSET_NAME);
    }

    JsonapiErrorMsg(GettextMsgPayload.Builder builder, String keyset) {
        this.payload = builder.build();
        this.keyset = keyset;
    }


    @Nonnull
    @Override
    public GettextMsgPayload getPayload() {
        return payload;
    }

    @Nonnull
    @Override
    public String getKeysetName() {
        return keyset;
    }

    public static JsonapiErrorMsg forStatus(HttpStatus httpStatus) {
        return STATUS_MESSAGE_MAP.get(httpStatus);
    }

    public static JsonapiErrorMsg forErrorId(Integer errorId) {
        return ERROR_ID_MESSAGE_MAP.get(errorId);
    }

    private static void fillStatusMessageMap() {
        STATUS_MESSAGE_MAP.put(HttpStatus.BAD_REQUEST, BAD_REQUEST);
        STATUS_MESSAGE_MAP.put(HttpStatus.NOT_ACCEPTABLE, NOT_ACCEPTABLE);
        STATUS_MESSAGE_MAP.put(HttpStatus.NOT_IMPLEMENTED, NOT_IMPLEMENTED);
        STATUS_MESSAGE_MAP.put(HttpStatus.TOO_MANY_REQUESTS, TOO_MANY_REQUESTS);
        STATUS_MESSAGE_MAP.put(HttpStatus.UNAUTHORIZED, UNAUTHORIZED);
        STATUS_MESSAGE_MAP.put(HttpStatus.UNSUPPORTED_MEDIA_TYPE, UNSUPPORTED_MEDIA_TYPE);
        STATUS_MESSAGE_MAP.put(HttpStatus.INTERNAL_SERVER_ERROR, INTERNAL_SERVER_ERROR);
        STATUS_MESSAGE_MAP.put(HttpStatus.NOT_FOUND, NOT_FOUND);
    }

    private static void fillErrorIdMessageMap() {
        ERROR_ID_MESSAGE_MAP.put(1, BAD_REQUEST);
        ERROR_ID_MESSAGE_MAP.put(8, NOT_ACCEPTABLE);
        ERROR_ID_MESSAGE_MAP.put(7, TOO_MANY_REQUESTS);
        ERROR_ID_MESSAGE_MAP.put(0, UNAUTHORIZED);
        ERROR_ID_MESSAGE_MAP.put(9, UNSUPPORTED_MEDIA_TYPE);
        ERROR_ID_MESSAGE_MAP.put(5, INTERNAL_SERVER_ERROR);
        ERROR_ID_MESSAGE_MAP.put(6, NOT_FOUND);
        ERROR_ID_MESSAGE_MAP.put(4, VALIDATION_ERROR);
    }

}
