package ru.yandex.partner.jsonapi.models;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesService;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.validation.AnnotationPathNodeConverterProvider;
import ru.yandex.partner.jsonapi.validation.CovariantPathNodeConverterProvider;

public abstract class AbstractApiModel<M extends ModelWithId> implements ApiModel<M> {
    private final ApiModelMetaData<M> apiModelMetaData;
    private final List<ApiField<M>> fields;
    private final Map<String, CrnkFilter<M, ?>> filters;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final Optional<DependsHolder> depends;
    private final ApiFieldsAccessRulesService<M> apiFieldsAvailableRules;
    private final ApiFieldsAccessRulesService<EditableData<M>> apiFieldsEditableRules;

    public AbstractApiModel(ApiModelMetaData<M> apiModelMetaData, List<ApiModelPart<M>> parts,
                            ApiFieldsAccessRulesService<M> apiFieldsAvailableRules,
                            ApiFieldsAccessRulesService<EditableData<M>> apiFieldsEditableRules) {
        this.apiModelMetaData = apiModelMetaData;
        this.apiFieldsAvailableRules = apiFieldsAvailableRules;
        this.apiFieldsEditableRules = apiFieldsEditableRules;
        var dependsHolders = parts.stream()
                .map(ApiModelPart::getDepends)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
        if (dependsHolders.isEmpty()) {
            depends = Optional.empty();
        } else {
            depends = Optional.of(new DependsHolder(dependsHolders.stream().map(DependsHolder::toMap)
                    .flatMap(m -> m.entrySet().stream())
                    .collect(Collectors.toMap(
                            Map.Entry::getKey,
                            Map.Entry::getValue,
                            (f, s) -> {
                                var res = new HashMap<>(f);
                                s.forEach((key, value) -> res.merge(key, value,
                                        (fv, sv) -> {
                                            var resVal = new ArrayList<String>(fv.size() + sv.size());
                                            resVal.addAll(fv);
                                            resVal.addAll(sv);
                                            return resVal;
                                        }));
                                return res;
                            }))));
        }
        this.fields = parts.stream()
                .map(ApiModelPart::getFields)
                .flatMap(List::stream)
                .collect(Collectors.toList());

        this.filters = parts.stream()
                .map(ApiModelPart::getFilters)
                .map(Map::entrySet)
                .flatMap(Set::stream)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        var builder = new CovariantPathNodeConverterProvider.Builder();
        parts.forEach(part -> part.configurePathNodeConverter(builder));
        this.pathNodeConverterProvider = builder
                .fallback(AnnotationPathNodeConverterProvider.getInstance())
                .build();
    }

    @Override
    public Optional<DependsHolder> getDepends() {
        return depends;
    }

    @Override
    public String getResourceType() {
        return apiModelMetaData.getResourceType();
    }

    @Override
    public Class<M> getModelClass() {
        return apiModelMetaData.getModelClass();
    }

    @Override
    public List<ApiField<M>> getFields() {
        return fields;
    }

    @Override
    public Map<String, CrnkFilter<M, ?>> getFilters() {
        return filters;
    }

    @Override
    public PathNodeConverterProvider getPathNodeConverterProvider() {
        return pathNodeConverterProvider;
    }

    @Override
    public ApiFieldsAccessRulesService<M> getApiFieldsAvailableRules() {
        return apiFieldsAvailableRules;
    }

    @Override
    public ApiFieldsAccessRulesService<EditableData<M>> getApiFieldsEditableRules() {
        return apiFieldsEditableRules;
    }
}
