package ru.yandex.partner.jsonapi.models;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.apache.commons.math3.util.Pair;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultConverters;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.partner.core.action.ActionContextWithModelProperty;
import ru.yandex.partner.core.action.ActionModelContainer;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.TransitionAction;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.factories.ActionFactory;
import ru.yandex.partner.core.action.factories.CustomPayloadActionFactory;
import ru.yandex.partner.core.action.factories.ModelPayloadActionFactory;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.UpdateLifecycle;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.core.validation.defects.presentation.CommonValidationMsg;
import ru.yandex.partner.jsonapi.crnk.DummyActionAllowedAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.authorization.request.RequestAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.exceptions.ValidationException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.crnk.fields.ModelPathForApi;
import ru.yandex.partner.jsonapi.crnk.fields.RequiredModelProperties;
import ru.yandex.partner.jsonapi.crnk.filter.AuthorizationFilterWrapper;
import ru.yandex.partner.jsonapi.crnk.filter.parser.FilterNode;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.jsonapi.models.actions.ActionService;
import ru.yandex.partner.jsonapi.models.relationships.ApiRelationship;
import ru.yandex.partner.libs.annotation.PartnerTransactional;
import ru.yandex.partner.libs.authorization.actioncontext.CreateRequest;
import ru.yandex.partner.libs.authorization.actioncontext.UpdateRequest;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.i18n.TranslatableError;

import static ru.yandex.partner.jsonapi.crnk.authorization.AuthorizationDecisionMatchers.permittedOrElseThrow;
import static ru.yandex.partner.jsonapi.crnk.filter.AuthorizationFilterWrapper.attachAuthorizationFilter;

public abstract class AbstractApiService<M extends ModelWithId> implements ApiService<M> {
    private static final String POST_RIGHT_NAME_TEMPLATE = "do_%s_add";

    protected final RequestAuthorizationService requestAuthorizationService;
    protected final ActionsAuthorizationService<M> actionsAuthorizationService;
    protected final ApiFieldsService<M> apiFieldsService;
    protected final ApiModel<M> apiModel;
    private final ActionService<M> actionService;
    private final Map<String, ApiRelationship<M>> apiRelationshipMap;
    private final DefectPresentationRegistry<TranslatableError> defectRegistry;
    private final ModelQueryService<? super M> modelService;
    private final Set<ModelProperty<? extends Model, ?>> propertiesForActions;
    private final ActionPerformer actionPerformer;
    private String defaultPostRightName;

    @SuppressWarnings("checkstyle:parameternumber")
    public AbstractApiService(
            DefectPresentationRegistry<TranslatableError> defectRegistry,
            ModelQueryService<? super M> modelService,
            RequestAuthorizationService requestAuthorizationService,
            ActionsAuthorizationService<M> actionsAuthorizationService,
            ApiFieldsService<M> apiFieldsService,
            ApiModel<M> apiModel,
            ActionService<M> actionService,
            ActionPerformer actionPerformer
    ) {
        this.defectRegistry = defectRegistry;
        this.modelService = modelService;
        this.requestAuthorizationService = requestAuthorizationService;
        this.actionsAuthorizationService = actionsAuthorizationService;
        this.apiFieldsService = apiFieldsService;
        this.apiModel = apiModel;
        this.apiRelationshipMap = apiModel.getApiRelationships().stream()
                .collect(Collectors.toMap(ApiRelationship::getField, Function.identity()));
        this.actionService = actionService;
        this.propertiesForActions = actionsAuthorizationService.getAllRequiredProperties()
                .stream()
                .map(property -> (ModelProperty<? extends Model, ?>) property)
                .collect(Collectors.toSet());
        this.actionPerformer = actionPerformer;
        this.defaultPostRightName = POST_RIGHT_NAME_TEMPLATE.formatted(apiModel.getResourceType());
    }

    public AbstractApiService(
            DefectPresentationRegistry<TranslatableError> defectRegistry,
            ModelQueryService<M> modelService,
            RequestAuthorizationService requestAuthorizationService,
            ApiFieldsService<M> apiFieldsService,
            ApiModel<M> apiModel,
            ActionService<M> actionService,
            ActionPerformer actionPerformer
    ) {
        this(
                defectRegistry,
                modelService,
                requestAuthorizationService,
                new DummyActionAllowedAuthorizationService<>(requestAuthorizationService, apiModel),
                apiFieldsService,
                apiModel,
                actionService,
                actionPerformer
        );
    }

    @Override
    public M findOne(Long id, Set<ModelProperty<? extends Model, ?>> modelProperties) {
        return findOne(id, modelProperties, false);
    }

    protected M findOne(Long id, Set<ModelProperty<? extends Model, ?>> modelProperties, boolean forUpdate) {
        List<M> list = modelService.findAll(QueryOpts.forClass(getModelClass())
                .withFilter(AuthorizationFilterWrapper.attachAuthorizationFilter(
                        CoreFilterNode.eq(apiModel.getIdFilter(), id),
                        getApiModel().getPolicy()
                                .authorizeGetRequest(getRequestAuthorization().currentRequestContext())
                ))
                .withProps(Sets.union(modelProperties, propertiesForActions))
                .forUpdate(forUpdate)
        );
        return list.isEmpty()
                ? null
                : list.get(0);
    }

    @Override
    public final List<M> findAll(Set<ModelProperty<? extends Model, ?>> modelProperties,
                                 @Nullable FilterNode filterNode,
                                 @Nullable LimitOffset limitOffset,
                                 @Nullable List<OrderBy> orderByList) {
        return findAll(modelProperties, filterNode, limitOffset, orderByList, false);
    }

    protected List<M> findAll(Set<ModelProperty<? extends Model, ?>> modelProperties,
                              @Nullable FilterNode filterNode,
                              @Nullable LimitOffset limitOffset,
                              @Nullable List<OrderBy> orderByList,
                              boolean forUpdate) {
        CoreFilterNode<M> coreFilterNode = filterNode == null
                ? CoreFilterNode.neutral()
                : filterNode.toCoreFilterNode(() -> getApiModel().getFilters());

        return findAll(modelProperties, coreFilterNode, limitOffset, orderByList, forUpdate);
    }

    protected List<M> findAll(Set<ModelProperty<? extends Model, ?>> modelProperties,
                              CoreFilterNode<M> coreFilterNode,
                              @Nullable LimitOffset limitOffset,
                              @Nullable List<OrderBy> orderByList,
                              boolean forUpdate) {
        coreFilterNode =
                AuthorizationFilterWrapper.attachAuthorizationFilter(coreFilterNode, getApiModel().getPolicy()
                        .authorizeGetAllRequest(getRequestAuthorization().currentRequestContext()));

        return modelService.findAll(QueryOpts.forClass(getModelClass())
                .withFilter(coreFilterNode)
                .withProps(Sets.union(modelProperties, propertiesForActions))
                .forUpdate(forUpdate)
                .withOrder(orderByList)
                .withLimitOffset(limitOffset)
        );
    }

    @PartnerTransactional
    @Override
    public M save(M model, IncomingApiFields<M> updatedFields, List<DefectInfo<Defect>> parseDefectInfos) {
        if (!parseDefectInfos.isEmpty()) {
            // set transaction for rollback due to parse errors, may continue work in dry mode
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }

        var result = saveInternal(model, updatedFields, !parseDefectInfos.isEmpty());

        return getResult(result, model.getId(), updatedFields, parseDefectInfos, true);
    }

    protected ActionsResult<?> saveInternal(M model, IncomingApiFields<M> updatedFields, boolean isDryRun) {
        return saveByAction(model, updatedFields, getUpdateRequestAuthLifecycle());
    }

    private ActionsResult<?> saveByAction(M model, IncomingApiFields<M> updatedFields,
                                          UpdateLifecycle<M> updateLifecycle) {
        return doActionInternal(
                getEditActionName(),
                model,
                updatedFields,
                updateLifecycle
        );
    }

    /**
     * Авторизационные проверки через механизм Policy, которые будут произведены после
     * вычитки текущего состояния модели из базы
     * (с достаточным наполнением для вызова логики проверки)
     */
    protected UpdateLifecycle<M> getUpdateRequestAuthLifecycle() {
        return (modelFromDb, newModelState) ->
                getApiModel().getPolicy().authorizeUpdateRequest(new UpdateRequest<>(
                        getRequestAuthorization().getUserAuthentication(),
                        modelFromDb,
                        newModelState
                )).match(permittedOrElseThrow());
    }

    @PartnerTransactional
    @Override
    public M create(M model, IncomingApiFields<M> incomingApiFields, List<DefectInfo<Defect>> parseDefectInfos) {
        if (!parseDefectInfos.isEmpty()) {
            // set transaction for rollback due to parse errors, may continue work in dry mode
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }

        Optional<M> modelFromDb = tryFindAlreadyExisting(model);

        if (modelFromDb.isPresent()) {
            model.setId(modelFromDb.get().getId());
            return save(model, incomingApiFields, parseDefectInfos);
        } else {
            return create0(model, incomingApiFields, parseDefectInfos);
        }
    }

    private M create0(M model, IncomingApiFields<M> incomingApiFields, List<DefectInfo<Defect>> parseDefectInfos) {
        Set<String> addAvailableFields =
                apiFieldsService.getAddFields(requestAuthorizationService.getAuthenticationFacade(), model);

        var unavailableFields = incomingApiFields.values()
                .stream()
                .map(ApiField::getJsonName)
                .filter(jsonName -> !addAvailableFields.contains(jsonName))
                .collect(Collectors.joining(", "));

        if (!unavailableFields.isEmpty()) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS,
                    MsgWithArgs.of(CommonValidationMsg.CANNOT_ADD_FIELDS, unavailableFields));
        }

        // Обогащение пропертями корневой модели
        // конечно хочется чтобы это происходило там же где и проперти субмодели кладутся
        // но пока это лишняя инфа в других ручках может и не стоит это делать
        Multimap<Model, ModelProperty<?, ?>> map = Multimaps.newSetMultimap(new IdentityHashMap<>(), HashSet::new);
        Collection<ApiField<M>> addedFields = incomingApiFields.values();
        for (ApiField<M> addedField : addedFields) {
            map.put(model, addedField.getModelPath().rootProperty());
        }
        incomingApiFields.addIncomingFields(map);

        authorizeCreateRequest(model);

        ActionsResult<?> result = createInternal(model, incomingApiFields);

        return getResult(result, parseDefectInfos);
    }

    protected ActionsResult<?> createInternal(M model, IncomingApiFields<M> incomingApiFields) {
        throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_IMPLEMENTED);
    }

    protected void authorizeCreateRequest(M model) {
        getApiModel().getPolicy().authorizeCreateRequest(new CreateRequest<>(
                getRequestAuthorization().getUserAuthentication(),
                model
        )).match(permittedOrElseThrow(HttpErrorStatusEnum.ERROR__NOT_FOUND));
        // Сервер должен возвращать 404 Not Found  в случае отсутствия у пользователя
        // прав на создание/редактирование сущности.
        // https://wiki.yandex-team.ru/partner/w/partner2-api/rest-json-api/#kodyotvetov
    }

    protected Optional<M> tryFindAlreadyExisting(M model) {
        return Optional.empty();
    }

    @Override
    public long count(@Nullable FilterNode filterNode) {
        CoreFilterNode<M> coreFilterNode = filterNode == null
                ? CoreFilterNode.neutral()
                : filterNode.toCoreFilterNode(() -> getApiModel().getFilters());

        coreFilterNode = attachAuthorizationFilter(coreFilterNode, getApiModel().getPolicy()
                .authorizeGetAllRequest(getRequestAuthorization().currentRequestContext()));
        return modelService.count(
                QueryOpts.forClass(getModelClass())
                        .withFilter(coreFilterNode));
    }

    protected DirtyChecking<M> getDirtyChecking() {
        return new AuthorizedFieldsDirtyChecking<>(
                apiFieldsService,
                apiModel,
                requestAuthorizationService
        );
    }

    protected DirtyChecking<M> getSimpleDirtyChecking() {
        return new SimpleDirtyChecking<>(
                apiFieldsService,
                apiModel
        );
    }

    protected <T extends Model> MassResult<String> convertMassResult(MassResult<T> result) {
        if (Objects.isNull(result)) {
            return MassResult.emptyMassAction();
        }

        return ResultConverters.<T, String>massResultValueConverter(
                        modelWithId -> Optional.ofNullable(modelWithId)
                                .filter(ModelWithId.class::isInstance)
                                .map(ModelWithId.class::cast)
                                .map(ModelWithId::getId)
                                .map(this::coreIdToApiId)
                                .orElse(null)
                )
                .convert(result);
    }

    protected void processApiErrors(Collection<MassResult<String>> allResults,
                                    @Nullable List<DefectInfo<Defect>> parseDefectInfos,
                                    @Nullable List<DefectInfo<Defect>> actionDefects) {
        List<DefectInfo<Defect>> defectInfos = new ArrayList<>();
        allResults.forEach(massResult -> {
            if (massResult != null && massResult.getValidationResult().hasAnyErrors()) {
                //так как элемент валидируется один, то в massResult только один элемент
                Result<String> result = massResult.toResultList().get(0);

                if (result == null) {
                    throw new IllegalStateException("Incorrect answer from core library");
                }


                PathNodeConverterProvider pathNodeConverterProvider = apiModel.getPathNodeConverterProvider();
                List<DefectInfo<Defect>> flattenedErrors = result.getValidationResult()
                        .flattenErrors(pathNodeConverterProvider);

                defectInfos.addAll(flattenedErrors);
            }
        });

        if (parseDefectInfos != null) {
            defectInfos.addAll(parseDefectInfos);
        }
        if (actionDefects != null) {
            defectInfos.addAll(actionDefects);
        }

        throwIfNotEmpty(defectInfos);
    }

    @Override
    public ApiModel<M> getApiModel() {
        return apiModel;
    }

    @Override
    public ApiFieldsService<M> getApiFieldsService() {
        return apiFieldsService;
    }

    @Override
    public boolean apiCanAdd() {
        if (postRightName() == null) {
            return true;
        }
        return requestAuthorizationService.userHasRight(postRightName());
    }

    @Override
    public boolean needToSetSelfLink() {
        return true;
    }

    @Nullable
    protected String postRightName() {
        return defaultPostRightName;
    }

    public RequestAuthorizationService getRequestAuthorization() {
        return requestAuthorizationService;
    }

    public ActionsAuthorizationService<M> getActionsAuthorizationService() {
        return actionsAuthorizationService;
    }

    public ModelQueryService<? super M> getModelService() {
        return modelService;
    }

    @Override
    public List<ApiRelationship<M>> getApiRelationships() {
        return apiModel.getApiRelationships();
    }

    @Override
    @Nullable
    public ApiRelationship<M> getApiRelationships(String fieldName) {
        return apiRelationshipMap.get(fieldName);
    }

    @Nullable
    @Override
    public ActionFactory<M, ?> getActionFactory(String actionName) {
        return actionService.getFactory(actionName);
    }

    @PartnerTransactional
    @Override
    public M doAction(String actionName,
                      M model,
                      IncomingApiFields<M> incomingApiFields,
                      List<DefectInfo<Defect>> parseDefectInfos) {
        if (!parseDefectInfos.isEmpty()) {
            // set transaction for rollback due to parse errors, may continue work in dry mode
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }

        var result = doActionInternal(
                actionName,
                model,
                incomingApiFields,
                (modelFromDb, newModelState) -> {
                }
        );

        return getResult(result, model.getId(), incomingApiFields, parseDefectInfos, true);
    }

    @PartnerTransactional
    @Override
    public M doAction(String actionName, M modelRef, JsonNode payload) {
        var result = doActionInternal(
                actionName,
                modelRef,
                payload
        );

        return getResult(result, modelRef.getId(), new IncomingApiFields<>(), null, true);
    }

    protected ActionsResult<?> doActionInternal(
            String actionName,
            M model,
            IncomingApiFields<M> updatedFields,
            UpdateLifecycle<M> updateLifecycle
    ) {
        checkCanDoAction(actionName);
        List<M> modelPayloads = List.of(model);
        ModelPayloadActionFactory<M, ?> actionFactory = actionService.getModelPayloadActionFactory(actionName);

        Map<Long, M> oldDataIndex = getOldDataIndexAndSaveToContext(actionFactory, modelPayloads, updatedFields);

        // first: api value, second: db value
        Map<Boolean, List<Pair<M, M>>> errorsPartition = modelPayloads.stream()
                .map(model1 -> new Pair<>(model1, oldDataIndex.get(model1.getId())))
                .collect(Collectors.partitioningBy(p -> p.getValue() != null));

        List<M> modelsWithoutErrors = errorsPartition.get(true).stream()
                .map(pair -> {
                    updateLifecycle.then(getActionAuthorizationLifecycle(actionName))
                            .beforeUpdate(pair.getValue(), pair.getKey());
                    return pair.getFirst();
                })
                .collect(Collectors.toList());

        if (!modelsWithoutErrors.isEmpty()) {
            List<ModelChanges<? super M>> modelChanges = prepareModelChanges(
                    modelsWithoutErrors, oldDataIndex, updatedFields
            );
            TransitionAction<M, ? extends ActionModelContainer<M>, ?> action =
                    actionFactory.createAction(modelChanges, updatedFields);

            ActionsResult<?> result = actionPerformer.doActions(action);
            addErrors(result, errorsPartition);
            return result;
        } else {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }
    }

    protected ActionsResult<?> doActionInternal(
            String actionName,
            M modelRef,
            JsonNode payload
    ) {
        checkCanDoAction(actionName);
        List<M> modelRefs = List.of(modelRef);
        CustomPayloadActionFactory<M, ?> factory = actionService.getCustomPayloadFactory(actionName);

        Map<Long, M> oldDataIndex = getOldDataIndexAndSaveToContext(factory, modelRefs, new IncomingApiFields<>());
        // first: api value, second: db value
        Map<Boolean, List<Pair<M, M>>> errorsPartition = modelRefs.stream()
                .map(model -> new Pair<>(model, oldDataIndex.get(model.getId())))
                .collect(Collectors.partitioningBy(p -> p.getValue() != null));

        List<M> modelsWithoutErrors = errorsPartition.get(true).stream()
                .map(pair -> {
                    getActionAuthorizationLifecycle(actionName).beforeUpdate(pair.getValue(), pair.getKey());
                    return pair.getSecond();
                })
                .collect(Collectors.toList());

        if (!modelsWithoutErrors.isEmpty()) {
            TransitionAction<M, ? extends ActionModelContainer<M>, ?> action =
                    factory.createAction(
                            modelsWithoutErrors.stream()
                                    .map(ModelWithId::getId)
                                    .collect(Collectors.toUnmodifiableList()),
                            payload
                    );
            ActionsResult<?> result = actionPerformer.doActions(action);
            addErrors(result, errorsPartition);
            return result;
        } else {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }
    }

    private void checkCanDoAction(String actionName) {
        String coreActionName = actionService.getCoreActionName(actionName);
        if (coreActionName == null || !actionsAuthorizationService.isActionExposed(coreActionName)) {
            throw new CrnkResponseStatusException(
                    HttpErrorStatusEnum.ERROR__PARAMS,
                    MsgWithArgs.of(JsonapiErrorMsg.CANNOT_DO_ACTION, actionName)
            );
        }
    }

    private Map<Long, M> getOldDataIndexAndSaveToContext(
            ActionFactory<M, ?> factory, List<M> models, IncomingApiFields<M> apiFields) {
        Set<ModelProperty<?, ?>> requiredFields = calculateRequiredFields(apiFields, factory);
        List<Long> ids = models.stream().map(ModelWithId::getId).collect(Collectors.toUnmodifiableList());

        ActionContextWithModelProperty<M, ?> context = actionPerformer.getActionContextFacade()
                .getActionContext(getModelClass());
        context.addFieldsToRequired(requiredFields);

        CoreFilterNode<M> filter = AuthorizationFilterWrapper.attachAuthorizationFilter(
                CoreFilterNode.neutral(),
                getApiModel().getPolicy().authorizeGetAllRequest(getRequestAuthorization().currentRequestContext())
        );

        List<M> oldData = context.getContainers(ids, requiredFields, true, filter).stream()
                .map(ActionModelContainer::getNonChangedItem)
                .collect(Collectors.toList());
        return oldData.stream().collect(Collectors.toMap(ModelWithId::getId, Function.identity()));
    }

    private void addErrors(ActionsResult<?> result, Map<Boolean, List<Pair<M, M>>> errorsPartition) {
        Map<Long, List<ActionError>> errors = errorsPartition.get(false).stream()
                .map(Pair::getFirst)
                .collect(Collectors.groupingBy(ModelWithId::getId))
                .entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey,
                        entry -> entry.getValue().stream()
                                .map(this::modelToActionError)
                                .collect(Collectors.toList())));
        result.addAllErrors(Map.of(getModelClass(), errors));
    }

    // todo хочется Set<ModelProperty<? super M, ?>>
    //  Тогда можно будет избавиться от метода каста в BlockActionAdd.prepareContext
    private Set<ModelProperty<?, ?>> calculateRequiredFields(
            IncomingApiFields<M> updatedFields,
            ActionFactory<M, ?> actionFactory) {
        Set<ModelProperty<?, ?>> requiredFields = new HashSet<>(
                actionsAuthorizationService.getRequiredPropertiesByAction(actionFactory::getName));

        //get incoming fields, requiredFields and editableFields
        for (String jsonName : updatedFields.getUpdatedFields()) {
            RequiredModelProperties requiredModelProperties =
                    apiFieldsService.getRequiredModelPropertiesByJsonName(jsonName);

            requiredFields.addAll(requiredModelProperties.getRequiredFields());
            requiredFields.addAll(requiredModelProperties.getEditableFields());
        }

        //get dependencies for action
        requiredFields.addAll(actionFactory.getActionConfiguration()
                .getDependsOnByClass(getModelClass()));

        requiredFields.addAll(
                apiFieldsService.getPropertiesForValidate(
                        updatedFields.values().stream()
                                .map(ApiField::getModelPath)
                                .filter(path -> !path.isNested())
                                .map(ModelPathForApi::rootProperty)
                                .collect(Collectors.toSet())
                )
        );

        return requiredFields;
    }

    private ActionError modelToActionError(M model) {
        return new ActionError(DefectInfoBuilder.of(CommonValidationMsg.ENTRIES_NOT_FOUND)
                .withValue(model.getId())
                .build(),
                ActionError.ActionDefectType.OTHER);
    }

    private List<ModelChanges<? super M>> prepareModelChanges(List<M> models, Map<Long, M> modelsFromDb,
                                                      IncomingApiFields<M> incomingFields) {
        List<ModelChanges<? super M>> modelChangesList = new ArrayList<>(modelsFromDb.size());
        for (M model : models) {
            M modelFromDb = modelsFromDb.get(model.getId());
            ModelChanges<M> modelChanges = getDirtyChecking().generateModelChanges(
                    modelFromDb, model, incomingFields);
            modelChangesList.add(modelChanges);
        }
        return modelChangesList;
    }

    private Class<M> getModelClass() {
        return getApiModel().getModelClass();
    }

    protected UpdateLifecycle<M> getActionAuthorizationLifecycle(String actionName) {
        String coreActionName = actionService.getCoreActionName(actionName);

        return (M modelFromDb, M newModelState) -> {
            if (coreActionName == null ||
                    !actionsAuthorizationService.checkActionAllowedAndAuthorized(coreActionName, modelFromDb)) {
                throw new CrnkResponseStatusException(
                        HttpErrorStatusEnum.ERROR__PARAMS,
                        MsgWithArgs.of(JsonapiErrorMsg.CANNOT_DO_ACTION, actionName)
                );
            }
        };
    }

    protected M getResult(ActionsResult<?> result, Long id,
                          IncomingApiFields<M> incomingApiFields,
                          @Nullable List<DefectInfo<Defect>> parseDefectInfos,
                          boolean refetch
    ) {
        var entityClass = getModelClass();
        MassResult<M> mr = result.getResult(entityClass).stream().findFirst().orElse(null);

        // если экшен не просто изменяет данные, а генерирует результат - нас интересует сам результат
        if (result.getResponseData() != null) {
            mr = (MassResult<M>) result.getResponseData();
            var idResult = mr.get(0);
            if (idResult.isSuccessful()) {
                id = idResult.getResult().getId();
            }
        }

        var allResults = StreamEx.of(result.getResults().values())
                .map(this::convertMassResult)
                .toList();

        var actionDefects = result.getErrors()
                .getOrDefault(entityClass, Collections.emptyMap())
                .getOrDefault(id, Collections.emptyList())
                .stream().filter(it -> !it.getDefectType().equals(ActionError.ActionDefectType.VALIDATION_DEFECT))
                .map(ActionError::getDefectInfo)
                .collect(Collectors.toList());

        processApiErrors(allResults, parseDefectInfos, actionDefects);

        if (refetch) {
            var allRequiredProperties = getActionsAuthorizationService().getAllRequiredProperties();

            var propsToRefetch = Sets.<ModelProperty<? extends Model, ?>>newHashSetWithExpectedSize(
                    allRequiredProperties.size() + incomingApiFields.keySet().size()
            );

            allRequiredProperties.forEach(propsToRefetch::add);

            propsToRefetch.addAll(
                    apiFieldsService.modelPropertiesFromRequestedFieldNames(incomingApiFields.keySet())
            );

            return findOne(id, propsToRefetch);
        } else {
            return mr.get(0).getResult();
        }
    }

    private M getResult(ActionsResult<?> result, List<DefectInfo<Defect>> parseDefectInfos) {
        var entityClass = getModelClass();
        MassResult<M> mr = result.getResult(entityClass).stream().findFirst().orElse(null);

        var allResults = StreamEx.of(result.getResults().values())
                .map(this::convertMassResult)
                .toList();

        var actionDefects = result.getErrors()
                .getOrDefault(entityClass, Collections.emptyMap())
                .values()
                .stream()
                .flatMap(List::stream)
                .filter(it -> !it.getDefectType().equals(ActionError.ActionDefectType.VALIDATION_DEFECT))
                .map(ActionError::getDefectInfo)
                .collect(Collectors.toList());

        processApiErrors(allResults, parseDefectInfos, actionDefects);

        return mr.get(0).getResult();
    }

    private void throwIfNotEmpty(List<DefectInfo<Defect>> defectInfos) {
        if (!defectInfos.isEmpty()) {
            List<TranslatableError> apiErrors = new ArrayList<>(defectInfos.size());
            defectInfos.forEach(d -> apiErrors.add(defectRegistry.getPresentation(d)));

            throw new ValidationException(apiErrors);
        }
    }

    protected Long getUserId() {
        return getRequestAuthorization().getUserAuthentication().getUid();
    }

    protected String getEditActionName() {
        return "edit";
    }
}
