package ru.yandex.partner.jsonapi.models;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.jsonapi.crnk.authorization.request.RequestAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.exceptions.ValidationException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.messages.JsonapiErrorMsg;
import ru.yandex.partner.libs.i18n.MsgWithArgs;
import ru.yandex.partner.libs.i18n.TranslatableError;

public class AuthorizedFieldsDirtyChecking<M extends ModelWithId> extends SimpleDirtyChecking<M> {
    private final ApiFieldsService<M> apiFieldsService;
    private final RequestAuthorizationService requestAuthorizationService;

    public AuthorizedFieldsDirtyChecking(
            ApiFieldsService<M> apiFieldsService,
            ApiModel<M> apiModel,
            RequestAuthorizationService requestAuthorizationService
    ) {
        super(apiFieldsService, apiModel);
        this.apiFieldsService = apiFieldsService;
        this.requestAuthorizationService = requestAuthorizationService;
    }

    @Override
    public ModelChanges<M> generateModelChanges(M modelFromDb, M modelFromApi, IncomingApiFields<M> updatedFields) {
        // Сгенерировали ModelChanges по пришедшим изменениям АПИ и старого базового объекта
        ModelChanges<M> modelChanges = super.generateModelChanges(modelFromDb, modelFromApi, updatedFields);

        M patchedModel = preparePatchedModel(modelFromDb, modelFromApi, modelChanges.getChangedPropsNames());

        Set<String> editableFields = apiFieldsService.getApiFieldsEditPermitRules()
                .calculate(requestAuthorizationService.getUserAuthentication(),
                        new EditableData<>(modelFromDb, patchedModel), updatedFields);

        Set<String> forbiddenFields = updatedFields.keySet().stream()
                .filter(jsonName -> !editableFields.contains(jsonName))
                .collect(Collectors.toSet());

        if (!forbiddenFields.isEmpty()) {
            MsgWithArgs msg = MsgWithArgs.of(
                    JsonapiErrorMsg.CANNOT_EDIT_FIELDS,
                    String.join(", ", forbiddenFields)
            );
            throw new ValidationException(List.of(new TranslatableError("/data/attributes/", msg)));
        }

        return modelChanges;
    }

    private M preparePatchedModel(M modelFromDb, M modelFromApi,
                                  Set<ModelProperty<? super M, ?>> changedModelProperties) {
        M model = apiFieldsService.getApiModel().createNewInstance();
        model.setId(modelFromDb.getId());

        var requiredModelProperties = apiFieldsService.getApiFieldsEditPermitRules().getRequiredModelProperties();

        for (ModelProperty<? extends Model, ?> requiredModelProperty : requiredModelProperties) {
            //noinspection unchecked
            var property = (ModelProperty<Model, Object>) requiredModelProperty;
            M modelFrom;
            if (changedModelProperties.contains(requiredModelProperty)) {
                modelFrom = modelFromApi;
            } else {
                modelFrom = modelFromDb;
            }

            property.set(model, property.get(modelFrom));
        }

        return model;
    }
}
