package ru.yandex.partner.jsonapi.models.actions;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.action.ActionPayloadType;
import ru.yandex.partner.core.action.TransitionAction;
import ru.yandex.partner.core.action.factories.ActionFactory;
import ru.yandex.partner.core.action.factories.CustomPayloadActionFactory;
import ru.yandex.partner.core.action.factories.ModelPayloadActionFactory;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.multistate.MultistateService;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;

@ParametersAreNonnullByDefault
public class ActionServiceImpl<M extends ModelWithId> implements ActionService<M> {
    private final Map<String, ActionFactory<M, ?>> actionCreatorMap;

    public ActionServiceImpl(List<ActionFactory<M, ? extends TransitionAction<M, ?, ?>>> actionFactories,
                             MultistateService multistateService) {
        this.actionCreatorMap = actionFactories.stream()
                .collect(Collectors.toMap(
                        actionFactory -> multistateService.getApiActionName(actionFactory.getName()),
                        Function.identity()));
    }

    @Override
    @Nullable
    public ActionFactory<M, ?> getFactory(String actionNAme) {
        return actionCreatorMap.get(actionNAme);
    }

    @Override
    public String getCoreActionName(String apiActionName) {
        ActionFactory<M, ?> actionFactory = actionCreatorMap.get(apiActionName);
        if (actionFactory == null) {
            return null;
        }
        return actionFactory.getName();
    }

    @Override
    public ModelPayloadActionFactory<M, ?> getModelPayloadActionFactory(String actionName) {
        ActionFactory<M, ?> actionFactory = getFactory(actionName);

        if (actionFactory == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }

        if (actionFactory.getType() != ActionPayloadType.MODEL) {
            throw new IllegalArgumentException(
                    "Unexpected ActionPayloadType. Expected: " + ActionPayloadType.MODEL
                            + ", Actual: " + actionFactory.getType());
        }
        try {
            //noinspection unchecked
            return (ModelPayloadActionFactory<M, ?>) actionFactory;
        } catch (ClassCastException e) {
            throw new IllegalArgumentException("ActionFactory with ActionPayloadType=" + ActionPayloadType.MODEL +
                    " must implement ModelPayloadActionFactory. Actual ActionFactory.class = "
                    + actionFactory.getClass());
        }
    }

    @Override
    public CustomPayloadActionFactory<M, ?> getCustomPayloadFactory(String actionName) {
        ActionFactory<M, ?> actionFactory = getFactory(actionName);

        if (actionFactory == null) {
            throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND);
        }

        if (actionFactory.getType() != ActionPayloadType.CUSTOM) {
            throw new IllegalArgumentException(
                    "Unexpected ActionPayloadType. Expected: " + ActionPayloadType.CUSTOM
                            + ", Actual: " + actionFactory.getType());
        }
        try {
            //noinspection unchecked
            return (CustomPayloadActionFactory<M, ?>) actionFactory;
        } catch (ClassCastException e) {
            throw new IllegalArgumentException("ActionFactory with ActionPayloadType=" + ActionPayloadType.CUSTOM +
                    " must implement CustomPayloadActionFactory. Actual ActionFactory.class = "
                    + actionFactory.getClass());
        }
    }
}
