package ru.yandex.partner.jsonapi.models.block;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplates;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.block.service.BlockAddService;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.page.model.PageWithOwner;
import ru.yandex.partner.jsonapi.crnk.BaseCrnkMetaInformation;
import ru.yandex.partner.jsonapi.crnk.authorization.actions.ActionsAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.authorization.request.RequestAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsService;
import ru.yandex.partner.jsonapi.crnk.fields.IncomingApiFields;
import ru.yandex.partner.jsonapi.models.AbstractApiService;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.jsonapi.models.block.actions.BlockActionService;
import ru.yandex.partner.libs.i18n.TranslatableError;

@ParametersAreNonnullByDefault
public class ApiBlockService<M extends BlockWithMultistate> extends AbstractApiService<M> {
    private final BlockAddService<BlockWithMultistate> blockAddService;
    private final AuthBlockContainerConfigurer<M, ?> authBlockContainerConfigurer;
    private final boolean hasDesignTemplates;
    private final ApiModelMetaData<? extends PageWithOwner> pageMetaData;

    @SuppressWarnings("checkstyle:parameternumber")
    public ApiBlockService(
            ActionPerformer actionPerformer,
            DefectPresentationRegistry<TranslatableError> defectRegistry,
            RequestAuthorizationService requestAuthorizationService,
            ActionsAuthorizationService<M> actionsAuthorizationService,
            ApiFieldsService<M> apiFieldsService,
            ApiModel<M> apiModel,
            ModelQueryService<BaseBlock> modelQueryService,
            BlockAddService<BlockWithMultistate> blockAddService,
            BlockActionService<M> blockActionService,
            AuthBlockContainerConfigurer<M, ?> authBlockContainerConfigurer,
            ApiModelMetaData<? extends PageWithOwner> pageMetaData) {
        super(
                defectRegistry,
                modelQueryService,
                requestAuthorizationService,
                actionsAuthorizationService,
                apiFieldsService,
                apiModel,
                blockActionService,
                actionPerformer
        );
        this.blockAddService = blockAddService;
        this.authBlockContainerConfigurer = authBlockContainerConfigurer;
        this.pageMetaData = pageMetaData;
        this.hasDesignTemplates = BlockWithDesignTemplates.class.isAssignableFrom(this.apiModel.getModelClass());
    }

    @Override
    protected ActionsResult<?> createInternal(M model, IncomingApiFields<M> incomingApiFields) {
        return blockAddService.addModels(
                getApiModel().getModelClass(),
                List.of(model),
                pageMetaData.getModelClass(),
                incomingApiFields,
                OperationMode.ADD,
                container -> authBlockContainerConfigurer.configureContainer(container, List.of(model))
        );
    }

    @Override
    public String coreIdToApiId(Long id) {
        return BlockUniqueIdConverter.convertToPublicId(id);
    }

    @Override
    public Long apiIdToCoreId(String id) {
        return BlockUniqueIdConverter.convertToUniqueId(id);
    }

    @Override
    public void fillMetaInformation(BaseCrnkMetaInformation baseCrnkMetaInformation) {
        if (hasDesignTemplates && authBlockContainerConfigurer.canValidateDesignAsManager()) {
            baseCrnkMetaInformation.setCanValidateDesignAsManager(true);
        }
    }
}
