package ru.yandex.partner.jsonapi.models.block;

import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.operation.container.OperationContainerConfigurer;
import ru.yandex.partner.jsonapi.crnk.page.PageReachabilityService;
import ru.yandex.partner.jsonapi.models.ApiModel;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;

public abstract class AuthBlockContainerConfigurer<B extends BaseBlock, P extends BasePage>
        implements OperationContainerConfigurer<BlockContainer> {
    private final AuthenticationFacade authenticationFacade;
    private final PageReachabilityService pageReachabilityService;
    private final ApiModel<B> blockApiModel;
    private final ApiModel<P> pageApiModel;

    public AuthBlockContainerConfigurer(
            AuthenticationFacade authenticationFacade,
            PageReachabilityService pageReachabilityService,
            ApiModel<B> blockApiModel,
            ApiModel<P> pageApiModel
    ) {
        this.authenticationFacade = authenticationFacade;
        this.pageReachabilityService = pageReachabilityService;
        this.blockApiModel = blockApiModel;
        this.pageApiModel = pageApiModel;
    }

    public boolean canValidateDesignAsManager() {
        return authenticationFacade.getUserAuthentication().userHasRight(blockApiModel
                .getResourceType().concat("_can_validate_design_as_manager"));
    }


    public boolean canEditRichMedia() {
        return authenticationFacade.getUserAuthentication().userHasRight(blockApiModel
                .getResourceType().concat("_edit_field__rich_media"));
    }

    private void addAuthAttributesToOperationContainer(BlockContainer blockContainer,
                                                       Collection<BaseBlock> preloadedModels) {

        CoreFilterNode<P> pagesFilter = pageReachabilityService.reachablePagesFilter(pageApiModel);
        Map<Class<? extends BaseBlock>, CoreFilterNode<P>> reachablePagesFilter = preloadedModels.stream()
                .filter(m -> m.getClass().isAssignableFrom(getModelClass()))
                .collect(Collectors.toMap(x -> x.getClass(), x -> pagesFilter, (a, b) -> b));
        blockContainer.getPageReachabilityFilter().putAll(reachablePagesFilter);

        boolean canValidateDesignAsManager = canValidateDesignAsManager();
        Map<Class<? extends BaseBlock>, Boolean> canValidateDesignsMap = preloadedModels.stream()
                .filter(m -> m.getClass().isAssignableFrom(getModelClass()))
                .collect(Collectors.toMap(x -> x.getClass(), x -> canValidateDesignAsManager, (a, b) -> b));
        blockContainer.getCanValidateDesignAsManager().putAll(canValidateDesignsMap);

        boolean canEditRichMedia = canEditRichMedia();
        Map<Class<? extends BaseBlock>, Boolean> canEditRichMediaMap = preloadedModels.stream()
                .filter(m -> m.getClass().isAssignableFrom(getModelClass()))
                .collect(Collectors.toMap(x -> x.getClass(), x -> canEditRichMedia, (a, b) -> b));
        blockContainer.getCanEditRichMedia().putAll(canEditRichMediaMap);
    }

    @Override
    public void configureContainer(BlockContainer container, Collection<BaseBlock> preloadedModels) {
        addAuthAttributesToOperationContainer(container, preloadedModels);
    }
}
