package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import io.crnk.core.engine.information.resource.ResourceFieldType;

import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.PageWithCommonFields;
import ru.yandex.partner.core.entity.page.model.prop.PageWithSiteDomainPropHolder;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.utils.OrderBy;
import ru.yandex.partner.jsonapi.crnk.CrnkModelFilters;
import ru.yandex.partner.jsonapi.crnk.authorization.request.RequestAuthorizationService;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.block.filter.parser.BlockIdValueParser;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.filter.parser.CrnkFilterParser;
import ru.yandex.partner.jsonapi.crnk.filter.parser.values.MatchCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.crnk.page.PageCrnkModelFilters;
import ru.yandex.partner.jsonapi.messages.block.BlockMsg;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.authorization.policy.base.Policy;

import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.partner.core.CoreConstants.LIMIT_CAMPAIGN_FOR_ADDING;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;

public abstract class ApiBaseBlockModelPart<B extends BaseBlock, P extends BasePage> implements ApiModelPart<B> {
    private static final String PAGE_FILTER = "page_filter";

    private final CrnkFilterParser crnkFilterParser;
    private final CrnkModelFilters<P> pageCrnkModelFilters;
    private final RequestAuthorizationService requestAuthorizationService;
    private final Policy<P> pagePolicy;
    private final BlockUniqueIdConverter.Prefixes prefix;
    private final ReachablePageService<B> reachablePageService;
    private final Supplier<CoreFilterNode<P>> authPageFilterProvider;
    private final Class<P> pageClass;
    private final boolean needPageIdFilter;

    @SuppressWarnings("checkstyle:parameternumber")
    public ApiBaseBlockModelPart(CrnkFilterParser crnkFilterParser,
                                 CrnkModelFilters<P> pageCrnkModelFilters,
                                 RequestAuthorizationService requestAuthorizationService,
                                 Policy<P> pagePolicy,
                                 BlockUniqueIdConverter.Prefixes prefix,
                                 ReachablePageService<B> reachablePageService,
                                 Supplier<CoreFilterNode<P>> authPageFilterProvider,
                                 Class<P> pageClass,
                                 boolean needPageIdFilter) {
        this.crnkFilterParser = crnkFilterParser;
        this.pageCrnkModelFilters = pageCrnkModelFilters;
        this.requestAuthorizationService = requestAuthorizationService;
        this.pagePolicy = pagePolicy;
        this.prefix = prefix;
        this.reachablePageService = reachablePageService;
        this.authPageFilterProvider = authPageFilterProvider;
        this.pageClass = pageClass;
        this.needPageIdFilter = needPageIdFilter;
    }

    @Override
    public List<ApiField<B>> getFields() {
        var pageIdFieldBuilder = ApiField.<B, Long>convertibleApiField(Long.class, B.PAGE_ID)
                .withAlwaysPermitAddFieldFunction()
                .withDefaultsFunction(pageIdDefaultsFunction())
                .withExtraQueryParamsConsumer(getExtraParamsConsumer());

        return List.of(
                ApiField.apiField(String.class, BatchFunction.<B, String>one(model -> null))
//                        Это поле совсем не нужно, оставил для соответствия тестов
//                        .withResourceFieldType(ResourceFieldType.ID)
//                      // Это поле просто для обозначения ResourceFieldType.ID, но не должно возвращаться в json
                        .withAvailableFunction(checkable(
                                new ApiFieldsAccessRulesFunction<>((userAuthentication, m) -> Boolean.FALSE)))
                        .build(BlockCrnkJsonFieldAliases.PUBLIC_ID),
                ApiField.<B, Long>convertibleApiField(Long.class, B.BLOCK_ID)
                        // Спецификация вынуждает пометить это поле как ID

//PreconditionUtil.verify(!jsonName.equals("id") || resourceFieldType == ResourceFieldType.ID,
//"only ID fields can be named 'id' for %s, consider adding @JsonApiId,
// ignoring it with @JsonIgnore or renaming it with @JsonProperty", resourceInformation);

                        // Не нашел пока мест, где это может заафектить,
                        // т.к. Resource собираем сами в DynamicResourceRepository
                        // Но не исключаю возможность, что такое может заафектить в будущих обновлениях
                        // Лучше бы переименовать это поле на block_id
                        // и тогда можно будет вернуть фиктивный field выше по коду

                        // Проверка живет тут ResourceFieldImpl.setResourceInformation
                        .withResourceFieldType(ResourceFieldType.ID)
                        .build(BlockCrnkJsonFieldAliases.ID),
                pageIdFieldBuilder
                        .build(BlockCrnkJsonFieldAliases.PAGE_ID)

        );
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        MatchCrnkFilterValueParser<BasePage> pageMatchCrnkFilterValueParser =
                new MatchCrnkFilterValueParser<>(crnkFilterParser,
                        requestAuthorizationService,
                        pageCrnkModelFilters,
                        pagePolicy);

        var list = List.<CrnkFilter<B, ?>>of(
                CrnkFilter.builder(BlockFilterNameEnum.ID)
                        .setLabel(RtbMsg.BLOCK_ID)
                        .setExposed(() -> true)
                        .toCrnkFilter(
                                BlockFilters.PUBLIC_ID,
                                new BlockIdValueParser(prefix)),
                CrnkFilter.builder(BlockFilterNameEnum.CAMPAIGN_ID)
                        .setLabel(RtbMsg.PAGE_ID)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.CAMPAIGN_ID),
                CrnkFilter.builder(BlockFilterNameEnum.PAGE_ID)
                        .setLabel(RtbMsg.PAGE_ID)
                        .setExposed(() -> false)
                        .toCrnkFilter(BlockFilters.PAGE_ID),
                CrnkFilter.builder(BlockFilterNameEnum.PAGE)
                        .toCrnkFilter(BlockFilters.PAGE, pageMatchCrnkFilterValueParser),
                CrnkFilter.builder(BlockFilterNameEnum.CAMPAIGN)
                        .toCrnkFilterMatch(
                                BlockFilters.PAGE,
                                pageMatchCrnkFilterValueParser,
                                CrnkFilter.builder(PageCrnkModelFilters.FilterNameEnum.ALL_DOMAIN)
                                        .setExposed(() -> true)
                                        .setLabel(BlockMsg.DOMAIN_AND_MIRROR))
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }

    private Consumer<QueryParamsContext<B>> getExtraParamsConsumer() {
        return context -> {
            Map<String, Set<String>> rawParams = context.getRawParams();

            CoreFilterNode<P> filter = null;
            if (rawParams.containsKey(PAGE_FILTER)) {
                var filterAsJsonString = getSingleFromQueryParams(rawParams, PAGE_FILTER);
                filter = crnkFilterParser.parse(filterAsJsonString)
                        .toCoreFilterNode(pageCrnkModelFilters);
            }

            context.putExtra(BlockCrnkJsonFieldAliases.PAGE_ID, CoreFilterNode.class, filter);

            LimitOffset limitOffset;
            try {
                limitOffset = limited(Integer.parseInt(getSingleFromQueryParams(rawParams, "page_limit")));
            } catch (RuntimeException e) {
                limitOffset = limited(LIMIT_CAMPAIGN_FOR_ADDING);
            }
            context.putExtra(BlockCrnkJsonFieldAliases.PAGE_ID, limitOffset.getClass(), limitOffset);

            Function<String, List<OrderBy>> converter = context.getCustomSortConverter();

            List<OrderBy> orderByList;
            try {
                orderByList = converter.apply(getSingleFromQueryParams(rawParams, "page_sort"));
            } catch (RuntimeException e) {
                orderByList = Collections.emptyList();
            }
            context.putExtra(BlockCrnkJsonFieldAliases.PAGE_ID, orderByList.getClass(), orderByList);
        };
    }

    // assume that request was parameters was like ?param=1 AND NOT like ?param=1&other=2&param=3
    private String getSingleFromQueryParams(Map<String, Set<String>> queryParams, String key) {
        Set<String> paramsForKey = queryParams.get(key);
        if (paramsForKey == null) {
            throw new NoSuchElementException(String.format("No %s in query params", key));
        }
        return paramsForKey.stream()
                .findFirst()
                .orElseThrow();
    }


    private Function<QueryParamsContext<B>, Map<String, Object>> pageIdDefaultsFunction() {
        return context -> {
            B block = context.getModelFromAttributes();
            CoreFilterNode<P> pageIdDefaultsFilter =
                    (CoreFilterNode<P>) context.getExtra(BlockCrnkJsonFieldAliases.PAGE_ID, CoreFilterNode.class);

            if (pageIdDefaultsFilter == null && block.getPageId() != null && needPageIdFilter) {
                pageIdDefaultsFilter = reachablePageService.getPageIdMetaFilter().eq(block.getPageId());
            }
            if (pageIdDefaultsFilter == null) {
                pageIdDefaultsFilter = CoreFilterNode.neutral();
            }

            return Map.of(
                    BlockCrnkJsonFieldAliases.PAGE_ID,
                    reachablePageService.getReachablePagesForAdd(
                                    authPageFilterProvider.get().and(pageIdDefaultsFilter),
                                    context.getExtra(BlockCrnkJsonFieldAliases.PAGE_ID, LimitOffset.class),
                                    Set.of(
                                            PageWithCommonFields.CAPTION,
                                            PageWithSiteDomainPropHolder.DOMAIN,
                                            BasePage.ID
                                    ), (List<OrderBy>) context.getExtra(BlockCrnkJsonFieldAliases.PAGE_ID, List.class),
                                    pageClass
                            ).stream()
                            .map(pageMapper())
                            .collect(Collectors.toList())
            );
        };
    }

    protected abstract Function<P, Map<String, Object>> pageMapper();
}
