package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.filter.description.provider.UserTypeDictionaryValuesProvider;
import ru.yandex.partner.jsonapi.crnk.filter.expose.CheckRightsCrnkFilterExposeStrategy;
import ru.yandex.partner.jsonapi.crnk.page.PageCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.page.PageCrnkMapper;
import ru.yandex.partner.jsonapi.crnk.page.PageRightsEnum;
import ru.yandex.partner.jsonapi.crnk.page.models.ContextPageForBlockCrnk;
import ru.yandex.partner.jsonapi.crnk.user.UserCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum;
import ru.yandex.partner.jsonapi.crnk.user.filter.parser.UserTypeCrnkFilterValueParser;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;

public class ApiCampaignModelPart<B extends BlockWithContextPage> implements ApiModelPart<B> {
    private final ApiModelMetaData<ContextPage> apiModelMetaData;
    private final PageCrnkMapper pageCrnkMapper;
    private final AuthenticationFacade authenticationFacade;

    public ApiCampaignModelPart(ApiModelMetaData<ContextPage> apiModelMetaData,
            PageCrnkMapper pageCrnkMapper, AuthenticationFacade authenticationFacade) {
        this.apiModelMetaData = apiModelMetaData;
        this.pageCrnkMapper = pageCrnkMapper;
        this.authenticationFacade = authenticationFacade;
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField
                        .readableApiField(
                                ContextPageForBlockCrnk.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, ContextPageForBlockCrnk>one(block ->
                                        pageCrnkMapper.contextPageToContextPageForBlockCrnk(block.getCampaign(),
                                                Set.of())))
                        .addCheckInnerField(
                                PageRightsEnum.VIEW_FIELD__DOMAIN_ID.getFullRightName(
                                        apiModelMetaData.getResourceType()),
                                PageCrnkJsonFieldAliases.DOMAIN_ID)
                        .addCheckInnerField(UserRightsEnum.VIEW_FIELD__IS_TUTBY,
                                PageCrnkJsonFieldAliases.OWNER,
                                UserCrnkJsonFieldAliases.IS_TUTBY)
                        .addCheckInnerField(UserRightsEnum.VIEW_FIELD__ROLES,
                                PageCrnkJsonFieldAliases.OWNER,
                                UserCrnkJsonFieldAliases.ROLES)
                        .build(BlockCrnkJsonFieldAliases.CAMPAIGN),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getBlocksCount()))
                        .build(BlockCrnkJsonFieldAliases.BLOCKS_COUNT),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getBlocksLimit()))
                        .build(BlockCrnkJsonFieldAliases.BLOCKS_LIMIT),
                ApiField
                        .readableApiField(
                                String.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, String>one(block -> block.getCampaign().getDomain()))
                        .build(BlockCrnkJsonFieldAliases.DOMAIN),
                ApiField
                        .readableApiField(Boolean.class, B.CAMPAIGN,
                                BatchFunction.one(this::isCampaignDeleted))
                        .build(BlockCrnkJsonFieldAliases.IS_CAMPAIGN_DELETED),
                ApiField
                        .readableApiField(Boolean.class, RtbBlock.CAMPAIGN,
                                BatchFunction.one(this::isCampaignDeleted))
                        .build(BlockCrnkJsonFieldAliases.IS_PAGE_DELETED),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getOwnerId()))
                        .build(BlockCrnkJsonFieldAliases.OWNER_ID),
                ApiField
                        .readableApiField(
                                String.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, String>one(block -> Optional.ofNullable(block)
                                        .map(B::getCampaign)
                                        .map(ContextPage::getOwner)
                                        .map(User::getLogin)
                                        .orElse(null)))
                        .withAvailableFunction(checkableByRight(
                                PageRightsEnum.VIEW_FIELD__LOGIN.getFullRightName(apiModelMetaData.getResourceType())))
                        .build(BlockCrnkJsonFieldAliases.LOGIN)
        );
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        CheckRightsCrnkFilterExposeStrategy.Factory exposeFactory
                = new CheckRightsCrnkFilterExposeStrategy.Factory(authenticationFacade);

        var list = List.<CrnkFilter<B, ?>>of(
                CrnkFilter.builder(BlockFilterNameEnum.LOGIN)
                        .setLabel(RtbMsg.LOGIN)
                        .setExposed(exposeFactory.forRights(UserRightsEnum.VIEW_SEARCH_FILTERS__LOGIN))
                        .toCrnkFilter(BlockFilters.LOGIN),

                CrnkFilter.builder(BlockFilterNameEnum.USER_TYPE)
                        .setLabel(RtbMsg.USER_TYPE)
                        .setExposed(exposeFactory.forRights(UserRightsEnum.VIEW_SEARCH_FILTERS__USER_TYPE))
                        .toCrnkFilter(
                                BlockFilters.USER_TYPE,
                                new UserTypeCrnkFilterValueParser(authenticationFacade),
                                new UserTypeDictionaryValuesProvider(authenticationFacade)
                        )
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }

    private Boolean isCampaignDeleted(B block) {
        var campaign = block.getCampaign();
        return campaign.getMultistate() == null
                ? null
                : campaign.getMultistate().test(PageStateFlag.DELETED);
    }
}
