package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Function;

import ru.yandex.partner.core.block.MobileBlockType;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockType;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockTypeAndCloseButtonDelay;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;

import static ru.yandex.partner.core.entity.block.type.closebuttondelay.BlockWithBlockTypeAndCloseButtonDelayModelProviderAndBkFiller.DEFAULT_CLOSE_BUTTON_DELAY_VALUE;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;

public class ApiCloseButtonDelayModelPart<B extends BlockWithBlockTypeAndCloseButtonDelay>
        implements ApiModelPart<B>, GodModeAvailable<B> {

    private final String resourceType;
    private final GodModeAvailable<B> godModeAvailableDelegate;

    public ApiCloseButtonDelayModelPart(String resourceType, GodModeAvailable<B> godModeAvailableDelegate) {
        this.resourceType = resourceType;
        this.godModeAvailableDelegate = godModeAvailableDelegate;
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField.<B, Long>convertibleApiField(Long.class, B.CLOSE_BUTTON_DELAY)
                        .withAvailableFunction(getCloseButtonDelayAvailableFunction())
                        .withEditableFunction(getCloseButtonDelayEditableFunction()) // same as
                        .withAddFieldFunction(getCloseButtonDelayEditableFunction()) // <-- !
                        .withDefaultsFunction(getCloseButtonDelayDefaultsFunction())
                        .build(BlockCrnkJsonFieldAliases.CLOSE_BUTTON_DELAY)
        );
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<B>> getCloseButtonDelayAvailableFunction() {
        return checkable(
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, block) -> {
                            boolean isInterstitial = MobileBlockType.INTERSTITIAL
                                    .getLiteral().equals(block.getBlockType());

                            return isInterstitial && userAuthentication.userHasRight(
                                        BlockRightNames.VIEW_FIELD__CLOSE_BUTTON_DELAY.getFullRightName(resourceType)
                            ) && userAuthentication.userHasRight(
                                    BlockRightNames.EDIT_FIELD__CLOSE_BUTTON_DELAY.getFullRightName(resourceType));
                        }, BlockWithBlockType.BLOCK_TYPE)
        );
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<B>>> getCloseButtonDelayEditableFunction() {
        return checkableAnd(List.of(
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, bEditableData) -> {
                            boolean isInterstitial = MobileBlockType.INTERSTITIAL
                                    .getLiteral().equals(bEditableData.getPatchedOrElseActual().getBlockType());

                            return isInterstitial && userAuthentication.userHasRight(
                                    BlockRightNames.EDIT_FIELD__CLOSE_BUTTON_DELAY.getFullRightName(resourceType)
                            );
                        }, BlockWithBlockType.BLOCK_TYPE),
                isGodModeEditableFunction()
        ));
    }

    private Function<QueryParamsContext<B>, Map<String, Object>> getCloseButtonDelayDefaultsFunction() {
        return context ->  {
            B block = context.getModelFromAttributes();

            if (Objects.nonNull(block.getBlockType()) &&
                    block.getBlockType().equals(MobileBlockType.INTERSTITIAL.getLiteral())) {
                return Map.of(
                        BlockCrnkJsonFieldAliases.CLOSE_BUTTON_DELAY, DEFAULT_CLOSE_BUTTON_DELAY_VALUE
                );
            }

            return Map.of();
        };
    }

    @Override
    public ApiFieldsAccessRulesFunction<B> isGodModeFunction() {
        return godModeAvailableDelegate.isGodModeFunction();
    }
}
