package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithCommonFields;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.AbstractApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.HasRightAccessRuleFunction;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.namedFunction;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;
import static ru.yandex.partner.jsonapi.models.FunctionNameEnum.GOD_MODE_FUNCTION;

public abstract class ApiCommonModelPart<B extends BlockWithCommonFields>
        extends AbstractApiModelPart<B> implements GodModeAvailable<B> {

    private final BlockRightNames editRight;

    public ApiCommonModelPart(String modelName) {
        this(modelName, BlockRightNames.EDIT_FIELD__COMMENT);
    }

    public ApiCommonModelPart(String modelName, BlockRightNames editRight) {
        super(modelName);
        this.editRight = editRight;
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField
                        .<B, String>convertibleApiField(String.class, B.CAPTION)
                        .withEditableFunction(namedFunction(GOD_MODE_FUNCTION))
                        .withAddFieldFunction(getAddFieldCaptionFunction())
                        .build(BlockCrnkJsonFieldAliases.CAPTION),
                ApiField
                        .<B, String>convertibleApiField(String.class, B.COMMENT)
                        .withAvailableFunction(checkableByRight(
                                BlockRightNames.VIEW_FIELD__COMMENT.getFullRightName(getModelName())))
                        .withEditableFunction(checkableAnd(List.of(
                                editableFunction(),
                                isGodModeEditableFunction())))
                        .withAddFieldFunction(checkableByRight(editRight.getFullRightName(getModelName())))
                        .build(BlockCrnkJsonFieldAliases.COMMENT)
        );
    }

    protected ApiFieldsAccessRulesFunction<EditableData<B>> editableFunction() {
        return new HasRightAccessRuleFunction<>(editRight.getFullRightName(getModelName()));
    }

    protected BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<B>>> getAddFieldCaptionFunction() {
        return ApiFieldsAccessRules.alwaysPermit();
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        var list = List.<CrnkFilter<B, ?>>of(
                CrnkFilter.builder(BlockFilterNameEnum.CAPTION)
                        .setLabel(RtbMsg.BLOCKS_CAPTION_ID)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.CAPTION)
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }
}
