package ru.yandex.partner.jsonapi.models.block.parts;

import java.io.IOException;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkData;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.models.AbstractApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.utils.PerlLikePrettyPrinter;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;

public class ApiCustomBkDataModelPart<B extends BlockWithCustomBkData> extends AbstractApiModelPart<B> {
    private final ObjectMapper mapper;

    public ApiCustomBkDataModelPart(String modelName) {
        super(modelName);

        mapper = new ObjectMapper();
        mapper.configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true);
        mapper.configure(SerializationFeature.INDENT_OUTPUT, true);
        mapper.setDefaultPrettyPrinter(new PerlLikePrettyPrinter());
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField
                        .<B, Boolean>convertibleApiField(Boolean.class, B.IS_CUSTOM_BK_DATA)
                        .withEditableFunction(checkableByRight(
                                BlockRightNames.EDIT_FIELD__BK_DATA.getFullRightName(getModelName())))
                        .build(BlockCrnkJsonFieldAliases.IS_CUSTOM_BK_DATA),
                ApiField
                        .convertibleApiField(String.class, B.BK_DATA,
                                BatchFunction.one(block -> {
                                    try {
                                        if (block.getBkData() == null) {
                                            return null;
                                        }
                                        final Object obj = mapper.readValue(block.getBkData(), Object.class);
                                        return mapper.writeValueAsString(obj);
                                    } catch (IOException e) {
                                        throw new IllegalStateException("Invalid json in bkData");
                                    }
                                }),
                                BatchBiFunction.<B, String>byModelProperty(B.BK_DATA))
                        .withAvailableFunction(checkableByRight(
                                BlockRightNames.VIEW_FIELD__BK_DATA.getFullRightName(getModelName())))
                        .withEditableFunction(checkable(new ApiFieldsAccessRulesFunction<>(
                                (userAuthentication, block) ->
                                        Boolean.TRUE.equals(block.getPatchedOrElseActual().getIsCustomBkData())
                                                && userAuthentication.userHasRight(
                                                BlockRightNames.EDIT_FIELD__BK_DATA.getFullRightName(getModelName())),
                                B.IS_CUSTOM_BK_DATA)))
                        .build(BlockCrnkJsonFieldAliases.BK_DATA)
        );
    }
}
