package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import one.util.streamex.StreamEx;

import ru.yandex.partner.core.entity.block.model.BlockWithDspsDspsUnmoderatedDspModeStrategy;
import ru.yandex.partner.core.entity.dsp.DspConstants;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.AbstractApiModelPart;
import ru.yandex.partner.jsonapi.models.ModelClassUtils;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.HasRightAccessRuleFunction;
import ru.yandex.partner.jsonapi.utils.ConvertingUtils;
import ru.yandex.partner.jsonapi.utils.function.BatchBiFunction;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;

public abstract class ApiDspsModelPart<B extends BlockWithDspsDspsUnmoderatedDspModeStrategy>
        extends AbstractApiModelPart<B> implements GodModeAvailable<B> {

    public ApiDspsModelPart(String modelName) {
        super(modelName);
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField
                        .convertibleApiField(ModelClassUtils.getListClass(String.class), B.DSPS,
                                BatchFunction.one(block -> {
                                    if (block.getDsps() == null) {
                                        return null;
                                    }
                                    return block.getDsps().stream()
                                            .map(blockDsp -> blockDsp.getId().toString())
                                            .collect(Collectors.toList());
                                }),
                                BatchBiFunction.<B, List<String>>one((block, strings) -> {
                                    if (strings == null) {
                                        return null;
                                    }

                                    var values = new ArrayList<Long>(strings.size());
                                    var defectInfos = ConvertingUtils.stringsToLong(strings, values::add);
                                    if (defectInfos.isEmpty()) {
                                        // NOTE: Здесь идёт преобразование в неполную модель Dsp, так как метод
                                        // вызывается при редактировании/добавлении и изменяет список Dsp на блоке
                                        block.setDsps(StreamEx.of(values)
                                                .map(id -> new Dsp().withId(id).withUnmoderatedRtbAuction(false))
                                                .toList());
                                        return null;
                                    } else {
                                        return defectInfos;
                                    }
                                }))
                        .withAvailableFunction(checkableByRight(
                                BlockRightNames.VIEW_FIELD__DSPS.getFullRightName(getModelName())))
                        .withEditableFunction(checkableAnd(List.of(
                                new HasRightAccessRuleFunction<>(
                                        BlockRightNames.EDIT_FIELD__DSPS.getFullRightName(getModelName())),
                                isGodModeEditableFunction(),
                                getDspsEditableFunction()
                        )))
                        .withAddFieldFunction(checkableByRight(BlockRightNames.EDIT_FIELD__DSPS
                                .getFullRightName(getModelName())))
                        .withDefaultsFunction(getDspsDefaultsFunction())
                        .build(BlockCrnkJsonFieldAliases.DSPS),
                ApiField
                        .readableApiField(
                                ModelClassUtils.getListClass(String.class),
                                B.DSPS,
                                BatchFunction.<B, List<String>>one(block ->
                                        getAllUnmoderatedDspsStream(block)
                                                .map(Object::toString)
                                                .collect(Collectors.toList())))
                        .withRequiredProperties(B.DSPS)
                        .withAvailableFunction(checkableAnd(List.of(
                                new HasRightAccessRuleFunction<>(BlockRightNames.
                                        VIEW_FIELD__ALL_UNMODERATED_DSPS.getFullRightName(getModelName())),
                                isUnmoderatedRtbAuctionFunction())))
                        .build(BlockCrnkJsonFieldAliases.ALL_UNMODERATED_DSPS),
                ApiField
                        .convertibleApiField(ModelClassUtils.getListClass(String.class),
                                B.DSPS_UNMODERATED,
                                BatchFunction.one(block -> block.getDspsUnmoderated()
                                        .stream()
                                        .filter(new Predicate<>() {
                                            private final Set<Long> allUnmoderatedDsps =
                                                    getAllUnmoderatedDspsStream(block).collect(Collectors.toSet());

                                            @Override
                                            public boolean test(Long aLong) {
                                                return allUnmoderatedDsps.contains(aLong);
                                            }
                                        })
                                        .map(Object::toString).collect(Collectors.toList())),
                                BatchBiFunction.<B, List<String>>one((block, strings) -> {
                                    if (strings == null) {
                                        return null;
                                    }
                                    var values = new ArrayList<Long>(strings.size());
                                    var defectInfos = ConvertingUtils.stringsToLong(strings, values::add);
                                    if (defectInfos.isEmpty()) {
                                        block.setDspsUnmoderated(values);
                                        return null;
                                    } else {
                                        return defectInfos;
                                    }
                                }))
                        .withRequiredProperties(B.DSPS)
                        .withAvailableFunction(checkable(isBlockDspsUnmoderatedAvailableFunction()))
                        .withEditableFunction(checkableAnd(List.of(
                                new HasRightAccessRuleFunction<>(
                                        BlockRightNames.EDIT_FIELD__UNMODERATED_DSPS.getFullRightName(getModelName())),
                                isGodModeEditableFunction(),
                                isBlockDspsUnmoderatedEditableFunction())))
                        .build(BlockCrnkJsonFieldAliases.UNMODERATED_DSPS)
        );
    }

    protected abstract ApiFieldsAccessRulesFunction<B> isUnmoderatedRtbAuctionFunction();

    protected abstract ApiFieldsAccessRulesFunction<B> isBlockDspsUnmoderatedAvailableFunction();

    protected abstract Function<QueryParamsContext<B>, Map<String, Object>> getDspsDefaultsFunction();

    private ApiFieldsAccessRulesFunction<EditableData<B>> isBlockDspsUnmoderatedEditableFunction() {
        var available = isBlockDspsUnmoderatedAvailableFunction();
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, bEditableModelData) -> available.getAccessRuleFunction()
                        .apply(userAuthentication, bEditableModelData.getPatchedOrElseActual()),

                available.getRequiredProperties()
        );
    }

    private Stream<Long> getAllUnmoderatedDspsStream(B block) {
        return Optional.ofNullable(block.getDsps())
                .map(values -> values.stream()
                        .filter(blockDsp ->
                                Boolean.TRUE.equals(blockDsp.getUnmoderatedRtbAuction()))
                        .sorted(Comparator.comparing(Dsp::getShortCaption))
                        .map(Dsp::getId))
                .orElse(Stream.empty());
    }

    private ApiFieldsAccessRulesFunction<EditableData<B>> getDspsEditableFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, bEditableData) ->
                !bEditableData.getPatchedOrElseActual().getDspMode().equals(DspConstants.DspMode.AUTO.getLiteral()),
                B.DSP_MODE);
    }
}
