package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.Collection;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockTypeAndShowVideo;
import ru.yandex.partner.core.entity.block.type.dspshowvideoandblocktype.MobileShowVideoApplicableService;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.models.AbstractApiModelPart;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.jsonapi.validation.CovariantPathNodeConverterProvider;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.sameAs;

/**
 * В этом парте регистрируются deprecated апи-филды,
 * содержимое которых игнорируется и не перекладывается в модель
 */
public class ApiIgnoredCompatFieldsModelPart<T extends BlockWithBlockTypeAndShowVideo> extends AbstractApiModelPart<T> {

    private final MobileShowVideoApplicableService mobileShowVideoApplicableService;
    private final List<ApiField<T>> fields;

    public ApiIgnoredCompatFieldsModelPart(
            String resourceType,
            MobileShowVideoApplicableService mobileShowVideoApplicableService,
            Collection<String> ignoredFields
    ) {
        super(resourceType);
        this.mobileShowVideoApplicableService = mobileShowVideoApplicableService;
        this.fields = makeIgnoredFields(ignoredFields);
    }

    private List<ApiField<T>> makeIgnoredFields(Collection<String> ignoredFields) {
        var firstField = ignoredFields.iterator().next();

        var firstApiField = ApiField.<T, Object, Long>readableApiField(Object.class, BaseBlock.ID,
                        BatchFunction.one(model -> null)
                )
                .withEditableFunction(notSimpleInappFunction())
                .withAvailableFunction(notSimpleInappAvailableFunction())
                .withAddFieldFunction(notSimpleInappFunction())
                .deprecated("deprecated mobile field")
                .build(firstField);

        return Stream.concat(
                Stream.of(firstApiField),
                ignoredFields.stream().skip(1).map(
                        fieldName -> (ApiField<T>) ApiField.<T, Object, Long>readableApiField(Object.class,
                                        BaseBlock.ID,
                                        BatchFunction.one(model -> null)
                                )
                                .withEditableFunction(sameAs(firstApiField))
                                .withAvailableFunction(sameAs(firstApiField))
                                .withAddFieldFunction(sameAs(firstApiField))
                                .deprecated("deprecated mobile field")
                                .build(fieldName)
                )
        ).collect(Collectors.toList());
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<T>>> notSimpleInappFunction() {
        return checkable(new ApiFieldsAccessRulesFunction<>(
                (ua, blockData) -> !mobileShowVideoApplicableService
                        .isSimpleInapp(blockData.getPatchedOrElseActual())
        ));
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<T>> notSimpleInappAvailableFunction() {
        return checkable(new ApiFieldsAccessRulesFunction<>(
                (ua, blockData) -> !mobileShowVideoApplicableService.isSimpleInappByCampaignOwner(blockData)
        ));
    }


    @Override
    public List<ApiField<T>> getFields() {
        return fields;
    }

    @Override
    public void configurePathNodeConverter(CovariantPathNodeConverterProvider.Builder builder) {
        // intentionally left empty
    }
}
