package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithInternalContextPage;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.crnk.page.PageCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.page.PageCrnkMapper;
import ru.yandex.partner.jsonapi.crnk.page.PageRightsEnum;
import ru.yandex.partner.jsonapi.crnk.page.models.InternalContextPageForBlockCrnk;
import ru.yandex.partner.jsonapi.crnk.user.UserCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.user.UserRightsEnum;
import ru.yandex.partner.jsonapi.messages.block.BlockMsg;
import ru.yandex.partner.jsonapi.models.ApiModelMetaData;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.auth.facade.AuthenticationFacade;

public class ApiInternalCampaignModelPart<B extends BlockWithInternalContextPage> implements ApiModelPart<B> {
    private final ApiModelMetaData<InternalContextPage> apiModelMetaData;
    private final PageCrnkMapper pageCrnkMapper;
    private final AuthenticationFacade authenticationFacade;

    public ApiInternalCampaignModelPart(ApiModelMetaData<InternalContextPage> apiModelMetaData,
                                        PageCrnkMapper pageCrnkMapper,
                                        AuthenticationFacade authenticationFacade) {
        this.apiModelMetaData = apiModelMetaData;
        this.pageCrnkMapper = pageCrnkMapper;
        this.authenticationFacade = authenticationFacade;
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        List<CrnkFilter<B, ?>> list = List.of(
                CrnkFilter.builder(BlockFilterNameEnum.MANAGER_LOGIN)
                        .setLabel(BlockMsg.MANAGER_LOGIN)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.MANAGER_LOGIN)
        );

        return list.stream()
                .collect(Collectors.toMap(CrnkFilter::getName, Function.identity()));
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField
                        .readableApiField(
                                InternalContextPageForBlockCrnk.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, InternalContextPageForBlockCrnk>one(block ->
                                        pageCrnkMapper.internalContextPageToInternalContextPageForBlockCrnk(
                                                block.getCampaign(), Set.of(),
                                                authenticationFacade.getUserAuthentication().getUid())))
                        .addCheckInnerField(
                                PageRightsEnum.VIEW_FIELD__DOMAIN_ID.getFullRightName(
                                        apiModelMetaData.getResourceType()),
                                PageCrnkJsonFieldAliases.DOMAIN_ID)
                        .addCheckInnerField(UserRightsEnum.VIEW_FIELD__IS_TUTBY,
                                PageCrnkJsonFieldAliases.OWNER,
                                UserCrnkJsonFieldAliases.IS_TUTBY)
                        .addCheckInnerField(UserRightsEnum.VIEW_FIELD__ROLES,
                                PageCrnkJsonFieldAliases.OWNER,
                                UserCrnkJsonFieldAliases.ROLES)
                        .build(BlockCrnkJsonFieldAliases.CAMPAIGN),
                ApiField
                        .readableApiField(
                                String.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, String>one(block -> block.getCampaign().getDomain()))
                        .build(BlockCrnkJsonFieldAliases.DOMAIN),
                ApiField
                        .readableApiField(Boolean.class, B.CAMPAIGN,
                                BatchFunction.one(this::isCampaignDeleted))
                        .build(BlockCrnkJsonFieldAliases.IS_CAMPAIGN_DELETED),
                ApiField
                        .readableApiField(Boolean.class, B.CAMPAIGN,
                                BatchFunction.one(this::isCampaignDeleted))
                        .build(BlockCrnkJsonFieldAliases.IS_PAGE_DELETED),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getBlocksCount()))
                        .build(BlockCrnkJsonFieldAliases.BLOCKS_COUNT),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getBlocksLimit()))
                        .build(BlockCrnkJsonFieldAliases.BLOCKS_LIMIT),
                ApiField
                        .readableApiField(
                                Long.class,
                                B.CAMPAIGN,
                                BatchFunction.<B, Long>one(block -> block.getCampaign().getOwnerId()))
                        .build(BlockCrnkJsonFieldAliases.OWNER_ID)
        );
    }

    private Boolean isCampaignDeleted(B block) {
        var campaign = block.getCampaign();
        return campaign.getMultistate() == null
                ? null
                : campaign.getMultistate().test(PageStateFlag.DELETED);
    }
}
