package ru.yandex.partner.jsonapi.models.block.parts;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithPlaceId;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;

import static ru.yandex.partner.core.entity.block.model.prop.BlockWithPlaceIdPlaceIdPropHolder.PLACE_ID;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkableByRight;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;

public class ApiPlaceIdModelPart<B extends BlockWithPlaceId> implements ApiModelPart<B>, GodModeAvailable<B> {

    private final String resourceType;
    private final GodModeAvailable<B> godModeAvailableDelegate;

    public ApiPlaceIdModelPart(String resourceType, GodModeAvailable<B> godModeAvailableDelegate) {
        this.resourceType = resourceType;
        this.godModeAvailableDelegate = godModeAvailableDelegate;
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField.<B, Long>convertibleApiField(Long.class, PLACE_ID)
                        .withAvailableFunction(
                                checkableByRight(BlockRightNames.VIEW_FIELD__PLACE_ID.getFullRightName(resourceType)))
                        .withEditableFunction(getEditableFunction())
                        .withAddFieldFunction(
                                checkableByRight(BlockRightNames.EDIT_FIELD__PLACE_ID.getFullRightName(resourceType))
                        )
                        .build(BlockCrnkJsonFieldAliases.PLACE_ID)
        );
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        var list = List.<CrnkFilter<B, ?>>of(
                CrnkFilter.builder(BlockFilterNameEnum.PLACE_ID)
                        .setLabel(RtbMsg.PLACE_ID)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.PLACE_ID)
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<EditableData<B>>> getEditableFunction() {
        return checkableAnd(List.of(
                new ApiFieldsAccessRulesFunction<>(
                        (userAuthentication, bEditableData) -> {
                            var patched = bEditableData.getPatched();
                            return (Objects.isNull(bEditableData.getActual().getPlaceId())
                                    || (patched != null && Objects.isNull(patched.getPlaceId())))
                                    && userAuthentication.userHasRight(
                                    BlockRightNames.EDIT_FIELD__PLACE_ID.getFullRightName(resourceType));
                        }, PLACE_ID
                ),
                isGodModeEditableFunction()
        ));
    }

    @Override
    public ApiFieldsAccessRulesFunction<B> isGodModeFunction() {
        return godModeAvailableDelegate.isGodModeFunction();
    }
}
