package ru.yandex.partner.jsonapi.models.block.parts;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithStrategy;
import ru.yandex.partner.core.entity.block.model.prop.BlockWithCommonShowVideoShowVideoPropHolder;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.block.filter.provider.StrategyDictionaryValueProvider;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.EditableData;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.AbstractApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.HasRightAccessRuleFunction;
import ru.yandex.partner.jsonapi.validation.CovariantPathNodeConverterProvider;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.alwaysPermit;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.checkable;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.neverPermit;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.sameAs;
import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction.checkableAnd;

@ParametersAreNonnullByDefault
public abstract class ApiStrategyModelPart<B extends
        BlockWithStrategy & BlockWithCommonShowVideoShowVideoPropHolder>
        extends AbstractApiModelPart<B> implements GodModeAvailable<B>, ShowVideoAvailable<B> {

    private final StrategyDictionaryValueProvider strategyDictionaryValueProvider;
    protected CovariantPathNodeConverterProvider converterProvider;

    public ApiStrategyModelPart(String modelName) {
        super(modelName);
        strategyDictionaryValueProvider = new StrategyDictionaryValueProvider();
    }

    public ApiStrategyModelPart(String modelName, StrategyDictionaryValueProvider strategyDictionaryValueProvider) {
        super(modelName);
        this.strategyDictionaryValueProvider = strategyDictionaryValueProvider;
    }

    @Override
    public List<ApiField<B>> getFields() {
        List<ApiFieldsAccessRulesFunction<EditableData<B>>> strategyEditableFunctionList = List.of(
                new HasRightAccessRuleFunction<>(
                        BlockRightNames.EDIT_FIELD_STRATEGY.getFullRightName(getModelName())),
                strategyEditableFunction(),
                isGodModeEditableFunction());
        var strategyField = ApiField
                .<B, Long>convertibleApiField(Long.class, B.STRATEGY_TYPE)
                .withEditableFunction(checkableAnd(strategyEditableFunctionList))
                .withAddFieldLikeEditableFunction()
                .withDefaultsFunction(getDefaultsFunction())
                .build(BlockCrnkJsonFieldAliases.STRATEGY);

        var videoActiveField = ApiField
                .<B, Boolean>convertibleApiField(Boolean.class, B.VIDEO_ACTIVE)
                .withAvailableFunction(checkable(videoActiveAvailableFunction()))
                .withEditableFunction(checkableAnd(List.of(
                        new HasRightAccessRuleFunction<>(
                                BlockRightNames.EDIT_FIELD_STRATEGY.getFullRightName(getModelName())),
                        isGodModeEditableFunction(),
                        new HasRightAccessRuleFunction<>(
                                BlockRightNames.EDIT_FIELD__SHOW_VIDEO.getFullRightName(getModelName())),
                        showVideoEditableFunction(),
                        hasVideoFieldsFunction()
                )))
                .withAddFieldFunction(checkableAnd(List.of(
                        new HasRightAccessRuleFunction<>(
                                BlockRightNames.EDIT_FIELD_STRATEGY.getFullRightName(getModelName())),
                        isGodModeEditableFunction(),
                        new HasRightAccessRuleFunction<>(
                                BlockRightNames.EDIT_FIELD__SHOW_VIDEO.getFullRightName(getModelName())),
                        showVideoAddFunction(),
                        isShowVideoApplicableFunction(),
                        hasVideoFieldsFunction()
                )))
                .build(BlockCrnkJsonFieldAliases.VIDEO_ACTIVE);
        var mediaActiveField = ApiField
                .<B, Boolean>convertibleApiField(Boolean.class, B.MEDIA_ACTIVE)
                .withEditableFunction(checkableAnd(Stream.concat(strategyEditableFunctionList.stream(),
                        Stream.of(getSeparateCPMAddFieldsFunction())).toList()))
                .withAddFieldFunction(checkableAnd(List.of(
                        new HasRightAccessRuleFunction<>(
                                BlockRightNames.EDIT_FIELD_STRATEGY.getFullRightName(getModelName())),
                        isGodModeEditableFunction(),
                        getSeparateCPMAddFieldsFunction())))
                .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                .build(BlockCrnkJsonFieldAliases.MEDIA_ACTIVE);

        return List.of(
                strategyField,
                mediaActiveField,
                ApiField
                        .<B, Boolean>convertibleApiField(Boolean.class, B.MEDIA_BLOCKED)
                        .withEditableFunction(sameAs(mediaActiveField))
                        .withAddFieldFunction(sameAs(mediaActiveField))
                        .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                        .build(BlockCrnkJsonFieldAliases.MEDIA_BLOCKED),
                ApiField
                        .<B, BigDecimal>convertibleApiField(BigDecimal.class, B.MEDIA_CPM)
                        .withEditableFunction(sameAs(mediaActiveField))
                        .withAddFieldFunction(sameAs(mediaActiveField))
                        .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                        .build(BlockCrnkJsonFieldAliases.MEDIA_CPM),
                ApiField
                        .<B, BigDecimal>convertibleApiField(BigDecimal.class, B.MINCPM)
                        .withEditableFunction(sameAs(strategyField))
                        .withAddFieldLikeEditableFunction()
                        .build(BlockCrnkJsonFieldAliases.MINCPM),
                ApiField
                        .<B, Boolean>convertibleApiField(Boolean.class, B.TEXT_ACTIVE)
                        .withEditableFunction(sameAs(mediaActiveField))
                        .withAddFieldFunction(sameAs(mediaActiveField))
                        .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                        .build(BlockCrnkJsonFieldAliases.TEXT_ACTIVE),
                ApiField
                        .<B, Boolean>convertibleApiField(Boolean.class, B.TEXT_BLOCKED)
                        .withEditableFunction(sameAs(mediaActiveField))
                        .withAddFieldFunction(sameAs(mediaActiveField))
                        .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                        .build(BlockCrnkJsonFieldAliases.TEXT_BLOCKED),
                ApiField
                        .<B, BigDecimal>convertibleApiField(BigDecimal.class, B.TEXT_CPM)
                        .withEditableFunction(sameAs(mediaActiveField))
                        .withAddFieldFunction(sameAs(mediaActiveField))
                        .withAvailableFunction(getSeparateCPMAvailaibleFieldsFunction())
                        .build(BlockCrnkJsonFieldAliases.TEXT_CPM),
                videoActiveField,
                ApiField
                        .<B, Boolean>convertibleApiField(Boolean.class, B.VIDEO_BLOCKED)
                        .withAvailableFunction(sameAs(videoActiveField))
                        .withEditableFunction(sameAs(videoActiveField))
                        .withAddFieldFunction(sameAs(videoActiveField))
                        .build(BlockCrnkJsonFieldAliases.VIDEO_BLOCKED),
                ApiField
                        .<B, BigDecimal>convertibleApiField(BigDecimal.class, B.VIDEO_CPM)
                        .withAvailableFunction(sameAs(videoActiveField))
                        .withEditableFunction(sameAs(videoActiveField))
                        .withAddFieldFunction(sameAs(videoActiveField))
                        .build(BlockCrnkJsonFieldAliases.VIDEO_CPM)
        );
    }

    protected boolean separateCpmSupported() {
        return true;
    }

    private BiConsumer<String, ApiFieldsAccessRules.Builder<B>> getSeparateCPMAvailaibleFieldsFunction() {
        return separateCpmSupported() ? alwaysPermit() : neverPermit();
    }

    protected ApiFieldsAccessRulesFunction<EditableData<B>> strategyEditableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, bEditableModelData) -> true);
    }

    protected ApiFieldsAccessRulesFunction<EditableData<B>> hasVideoFieldsFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, bEditableModelData) -> true);
    }

    private ApiFieldsAccessRulesFunction<EditableData<B>> getSeparateCPMAddFieldsFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, bEditableModelData) -> separateCpmSupported());
    }


    protected abstract Function<QueryParamsContext<B>, Map<String, Object>> getDefaultsFunction();

    protected Map<Long, GettextMsg> getDictionaryValues() {
        return strategyDictionaryValueProvider.getValues();
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        String strategyParameters = "strategy_parameters";
        var list = List.<CrnkFilter<B, ?>>of(
                // Strategy Filters
                CrnkFilter.builder(BlockFilterNameEnum.STRATEGY)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.STRATEGY)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.STRATEGY,
                                strategyDictionaryValueProvider),

                CrnkFilter.builder(BlockFilterNameEnum.MIN_CPM)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.MINIMUM_CPM)
                        .setExposed(() -> true)
                        .toCrnkFilter(BlockFilters.MIN_CPM),

                CrnkFilter.builder(BlockFilterNameEnum.TEXT_CPM)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.TEXT_CPM)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.TEXT_CPM),

                CrnkFilter.builder(BlockFilterNameEnum.TEXT_BLOCKED)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.TEXT_BLOCKED)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.TEXT_BLOCKED),

                CrnkFilter.builder(BlockFilterNameEnum.MEDIA_CPM)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.MEDIA_CPM)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.MEDIA_CPM),

                CrnkFilter.builder(BlockFilterNameEnum.MEDIA_BLOCKED)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.MEDIA_BLOCKED)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.MEDIA_BLOCKED),

                CrnkFilter.builder(BlockFilterNameEnum.VIDEO_CPM)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.VIDEO_CPM)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.VIDEO_CPM),

                CrnkFilter.builder(BlockFilterNameEnum.VIDEO_BLOCKED)
                        .setGroup(strategyParameters)
                        .setLabel(RtbMsg.VIDEO_BLOCKED)
                        .setExposed(this::separateCpmSupported)
                        .toCrnkFilter(BlockFilters.VIDEO_BLOCKED)
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }

    protected abstract ApiFieldsAccessRulesFunction<B> videoActiveAvailableFunction();
}
