package ru.yandex.partner.jsonapi.models.block.relationships;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import io.crnk.core.engine.document.Relationship;
import io.crnk.core.engine.document.ResourceIdentifier;
import io.crnk.core.utils.Nullable;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithPageWithOwner;
import ru.yandex.partner.core.entity.page.model.PageWithOwner;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.dbmeta.models.PublicIdFilterModel;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.models.ApiResourceType;
import ru.yandex.partner.jsonapi.models.relationships.ManyApiRelationship;

// Класс оставлен как пример релазации ManyApiRelationship
public class TestManyApiRelationship<T extends BlockWithPageWithOwner> implements ManyApiRelationship<T> {
    private final ModelQueryService<? super T> queryService;
    private final Class<T> blockClass;
    private final BlockUniqueIdConverter.Prefixes blockPrefix;
    private final ModelProperty<? super T, ? extends PageWithOwner> pageProp;

    public TestManyApiRelationship(
            ModelQueryService<? super T> queryService,
            Class<T> blockClass,
            BlockUniqueIdConverter.Prefixes blockPrefix,
            ModelProperty<? super T, ? extends PageWithOwner> pageProp
    ) {
        this.queryService = queryService;
        this.blockClass = blockClass;
        this.blockPrefix = blockPrefix;
        this.pageProp = pageProp;
    }

    @Override
    public List<String> getFieldsForCalcId() {
        return List.of(BlockCrnkJsonFieldAliases.CAMPAIGN);
    }

    @Override
    public ApiResourceType getApiResourceType() {
        return ApiResourceType.USERS;
    }

    @Override
    public String getField() {
        return "many_users_field_name";
    }

    @Override
    public Relationship getRelationship(T model) {
        var relationship = new Relationship();
        relationship.setData(Nullable.of(
                List.of(
                        new ResourceIdentifier(Long.toString(model.getCampaign().getOwnerId()),
                                getApiResourceType().getResourceType())
                )
        ));
        return relationship;
    }

    @Override
    public Map<String, List<String>> getSourceIdToTargetIdsMap(Collection<String> sourceIds) {
        return queryService.findAll(QueryOpts.forClass(blockClass)
                        .withFilter(CoreFilterNode.create(BlockFilters.PUBLIC_ID,
                                FilterOperator.IN,
                                sourceIds.stream()
                                        .map(id -> new PublicIdFilterModel(
                                                blockPrefix, id
                                        ))
                                        .collect(Collectors.toList())
                        ))
                .withProps(Set.of(pageProp, BaseBlock.PAGE_ID, BaseBlock.ID)))
                .stream()
                .collect(Collectors.groupingBy(
                        block -> BlockUniqueIdConverter.convertToPublicId(
                                blockPrefix,
                                block.getPageId(),
                                block.getBlockId()
                        ),
                        Collectors.mapping(block -> block.getCampaign().getOwnerId().toString(),
                                Collectors.toList())
                ));
    }
}
