package ru.yandex.partner.jsonapi.models.block.relationships;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import io.crnk.core.engine.document.Relationship;
import io.crnk.core.engine.document.ResourceIdentifier;
import io.crnk.core.utils.Nullable;

import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.model.BlockWithPageWithOwner;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.dbmeta.models.PublicIdFilterModel;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.models.ApiResourceType;
import ru.yandex.partner.jsonapi.models.relationships.OneApiRelationship;

// Класс оставлен как пример релазации OneApiRelationship
public class TestOneApiRelationship<T extends BlockWithPageWithOwner> implements OneApiRelationship<T> {
    private final ModelQueryService<? super T> queryService;
    private final Class<T> blockClass;
    private final BlockUniqueIdConverter.Prefixes blockPrefix;

    public TestOneApiRelationship(
            ModelQueryService<? super T> queryService,
            Class<T> blockClass,
            BlockUniqueIdConverter.Prefixes blockPrefix) {
        this.queryService = queryService;
        this.blockClass = blockClass;
        this.blockPrefix = blockPrefix;
    }

    @Override
    public List<String> getFieldsForCalcId() {
        return List.of(BlockCrnkJsonFieldAliases.CAMPAIGN);
    }

    @Override
    public ApiResourceType getApiResourceType() {
        return ApiResourceType.USERS;
    }

    @Override
    public String getField() {
        return "one_users_field_name";
    }

    @Override
    public Relationship getRelationship(T model) {
        var relationship = new Relationship();
        relationship.setData(
                Nullable.of(new ResourceIdentifier(Long.toString(model.getCampaign().getOwnerId()),
                        getApiResourceType().getResourceType()))
        );
        return relationship;
    }

    @Override
    public Map<String, String> getSourceIdToTargetIdMap(Collection<String> sourceIds) {
        return queryService.findAll(QueryOpts.forClass(blockClass)
                        .withFilter(CoreFilterNode.create(BlockFilters.PUBLIC_ID,
                                FilterOperator.IN,
                                sourceIds.stream()
                                        .map(id -> new PublicIdFilterModel(
                                                blockPrefix, id
                                        ))
                                        .collect(Collectors.toList())
                        ))
                        .withProps(Set.of(
                                BlockWithContextPage.CAMPAIGN,
                                BlockWithPageWithOwner.PAGE_ID,
                                BlockWithPageWithOwner.ID
                        ))
                )
                .stream()
                .collect(Collectors.toMap(
                        block -> BlockUniqueIdConverter.convertToPublicId(
                                blockPrefix,
                                block.getPageId(),
                                block.getBlockId()
                        ),
                        block -> block.getCampaign().getOwnerId().toString()));
    }
}
