package ru.yandex.partner.jsonapi.models.block.rtb.external;

import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.dsp.service.DspService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiDspsModelPart;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;


public class ApiRtbDspsModelPart extends ApiDspsModelPart<RtbBlock> {

    private final DspService dspService;
    private final ReachablePageService<RtbBlock> reachablePageService;

    public ApiRtbDspsModelPart(String modelName, DspService dspService,
                               ReachablePageService<RtbBlock> reachablePageService) {
        super(modelName);
        this.dspService = dspService;
        this.reachablePageService = reachablePageService;
    }

    @Override
    protected ApiFieldsAccessRulesFunction<RtbBlock> isUnmoderatedRtbAuctionFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, rtbBlock) ->
                Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                BlockWithContextPage.CAMPAIGN
        );
    }

    @Override
    protected ApiFieldsAccessRulesFunction<RtbBlock> isBlockDspsUnmoderatedAvailableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, rtbBlock) ->
                        userAuthentication.userHasRight(
                                BlockRightNames.VIEW_FIELD__UNMODERATED_DSPS.getFullRightName(getModelName()))
                                && rtbBlock.getCampaign() != null
                                && Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                BlockWithContextPage.CAMPAIGN);
    }

    @Override
    public ApiFieldsAccessRulesFunction<RtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Function<QueryParamsContext<RtbBlock>, Map<String, Object>> getDspsDefaultsFunction() {
        return context -> {
            RtbBlock block = context.getModelFromAttributes();
            ContextPage campaign = Optional.ofNullable(block.getPageId())
                    .map(PageFilters.PAGE_ID::eq)
                    .map(filter -> reachablePageService.getReachablePages(filter, ContextPage.class))
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS))
                    .stream().findFirst()
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND));
            block.setCampaign(campaign);

            UserAuthentication user = context.getUserAuthentication();
            boolean userHasRight = user.userHasRight(BlockRightNames.VIEW_FIELD__DSPS.getFullRightName(getModelName()));
            boolean canEditRichMedia = user.userHasRight(BlockRightNames.EDIT_FIELD__RICH_MEDIA
                    .getFullRightName(getModelName()));
            Boolean pageIsUnmoderatedAuction = block.getCampaign().getUnmoderatedRtbAuction();
            return dspService.getDefaults(block, userHasRight, pageIsUnmoderatedAuction, canEditRichMedia);
        };
    }
}
