package ru.yandex.partner.jsonapi.models.block.rtb.external;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.context.MessageSource;

import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.block.filter.SiteVersionValueParserAndProvider;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiSiteVersionModelPart;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;

public class ApiRtbSiteVersionModelPart extends ApiSiteVersionModelPart<RtbBlock> {
    private final SiteVersionAvailabilityService siteVersionAvailabilityService;
    private final ReachablePageService<RtbBlock> pageService;

    public ApiRtbSiteVersionModelPart(
            String modelName,
            MessageSource messageSource,
            SiteVersionAvailabilityService siteVersionAvailabilityService,
            ReachablePageService<RtbBlock> pageService) {
        super(
                modelName,
                new SiteVersionValueParserAndProvider(
                        messageSource,
                        siteVersionAvailabilityService.getAvailableVersions(RtbBlock.class)
                )
        );
        this.siteVersionAvailabilityService = siteVersionAvailabilityService;
        this.pageService = pageService;
    }

    @Override
    public ApiFieldsAccessRulesFunction<RtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Map<String, Object> getSiteVersionDefaults(
            QueryParamsContext<RtbBlock> context) {
        RtbBlock block = context.getModelFromAttributes();
        var filter = Optional.ofNullable(block.getPageId())
                .map(pageId -> CoreFilterNode.eq(PageFilters.PAGE_ID, pageId))
                .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS));

        List<ContextPage> pages = pageService.getReachablePages(
                filter,
                null,
                Set.of(ContextPage.ALLOWED_AMP, ContextPage.ALLOWED_TURBO, ContextPage.OWNER),
                null,
                ContextPage.class
        );
        ContextPage page = pages.stream().findFirst()
                .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS));

        Set<String> features = new HashSet<>(page.getOwner().getFeatures());

        var dependedVersions = siteVersionAvailabilityService
                .getSiteVersionsThatDependOnFeature(block.getClass(), features);

        Set<String> availableVersions = siteVersionAvailabilityService.getBlockSiteVersions(
                page,
                dependedVersions,
                dependedVersions.contains(SiteVersionType.TURBO_DESKTOP.getLiteral()),
                block.getClass()
        );

        return Map.of(
                BlockCrnkJsonFieldAliases.SITE_VERSION,
                availableVersions.stream()
                        .map(version -> Map.of(
                                "id", version,
                                "label", getSiteVersionLabel(version)
                        ))
                        .collect(Collectors.toList())
        );
    }
}
