package ru.yandex.partner.jsonapi.models.block.rtb.internal;

import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.dsp.service.DspService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiDspsModelPart;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;


public class ApiInternalRtbDspsModelPart extends ApiDspsModelPart<InternalRtbBlock> {

    private final DspService dspService;
    private final ReachablePageService<InternalRtbBlock> reachablePageService;

    public ApiInternalRtbDspsModelPart(String modelName, DspService dspService,
                                       ReachablePageService<InternalRtbBlock> reachablePageService) {
        super(modelName);
        this.dspService = dspService;
        this.reachablePageService = reachablePageService;
    }

    @Override
    protected ApiFieldsAccessRulesFunction<InternalRtbBlock> isUnmoderatedRtbAuctionFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, rtbBlock) ->
                Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                InternalRtbBlock.CAMPAIGN
        );
    }

    @Override
    protected ApiFieldsAccessRulesFunction<InternalRtbBlock> isBlockDspsUnmoderatedAvailableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, rtbBlock) ->
                        userAuthentication.userHasRight(
                                BlockRightNames.VIEW_FIELD__UNMODERATED_DSPS.getFullRightName(getModelName()))
                                && rtbBlock.getCampaign() != null
                                && Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                InternalRtbBlock.CAMPAIGN);
    }

    @Override
    public ApiFieldsAccessRulesFunction<InternalRtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Function<QueryParamsContext<InternalRtbBlock>, Map<String, Object>> getDspsDefaultsFunction() {
        return context -> {
            InternalRtbBlock block = context.getModelFromAttributes();

            InternalContextPage campaign = Optional.ofNullable(block.getPageId())
                    .map(pageId -> CoreFilterNode.eq(PageFilters.PAGE_ID, pageId))
                    .map(filter -> reachablePageService.getReachablePages(filter, InternalContextPage.class))
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS))
                    .stream().findFirst()
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND));
            block.setCampaign(campaign);

            UserAuthentication user = context.getUserAuthentication();

            boolean userHasRight = user.userHasRight(BlockRightNames.VIEW_FIELD__DSPS.getFullRightName(getModelName()));
            boolean userCanEditRichMedia = user.userHasRight(BlockRightNames.EDIT_FIELD__RICH_MEDIA
                    .getFullRightName(getModelName()));
            Boolean pageIsUnmoderatedAuction = block.getCampaign().getUnmoderatedRtbAuction();
            return dspService.getDefaults(block, userHasRight, pageIsUnmoderatedAuction, userCanEditRichMedia);
        };
    }
}
