package ru.yandex.partner.jsonapi.models.block.rtb.internal;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.context.MessageSource;

import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.entity.page.model.PageWithAllowedTurbo;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.block.filter.SiteVersionValueParserAndProvider;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiSiteVersionModelPart;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;

public class ApiInternalRtbSiteVersionModelPart extends ApiSiteVersionModelPart<InternalRtbBlock> {
    private final SiteVersionAvailabilityService siteVersionAvailabilityService;
    private final ReachablePageService<InternalRtbBlock> pageService;

    public ApiInternalRtbSiteVersionModelPart(
            String modelName,
            MessageSource messageSource,
            SiteVersionAvailabilityService siteVersionAvailabilityService,
            ReachablePageService<InternalRtbBlock> pageService) {
        super(
                modelName,
                new SiteVersionValueParserAndProvider(
                        messageSource,
                        siteVersionAvailabilityService.getAvailableVersions(InternalRtbBlock.class)
                )
        );
        this.siteVersionAvailabilityService = siteVersionAvailabilityService;
        this.pageService = pageService;
    }

    @Override
    public ApiFieldsAccessRulesFunction<InternalRtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Map<String, Object> getSiteVersionDefaults(
            QueryParamsContext<InternalRtbBlock> context) {
        InternalRtbBlock block = context.getModelFromAttributes();
        var filter = Optional.ofNullable(block.getPageId())
                .map(pageId -> CoreFilterNode.eq(PageFilters.PAGE_ID, pageId))
                .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS));

        List<InternalContextPage> pages = pageService.getReachablePages(
                filter,
                null,
                Set.of(PageWithAllowedTurbo.ALLOWED_TURBO, InternalContextPage.OWNER),
                null,
                InternalContextPage.class
        );
        InternalContextPage page = pages.stream().findFirst()
                .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS));

        Set<String> features = new HashSet<>(page.getOwner().getFeatures());

        var dependedVersions = siteVersionAvailabilityService
                .getSiteVersionsThatDependOnFeature(block.getClass(), features);

        Set<String> availableVersions = siteVersionAvailabilityService
                .getBlockSiteVersions(page, dependedVersions, true, block.getClass());

        return Map.of(
                BlockCrnkJsonFieldAliases.SITE_VERSION,
                availableVersions.stream()
                        .map(version -> Map.of(
                                "id", version,
                                "label", getSiteVersionLabel(version)
                        ))
                        .collect(Collectors.toList())
        );
    }
}

