package ru.yandex.partner.jsonapi.models.block.rtb.mobile;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.context.MessageSource;
import org.springframework.context.support.MessageSourceAccessor;

import ru.yandex.partner.core.block.MobileBlockType;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockType;
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BlockValidationMsg;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.entity.page.service.validation.defects.presentation.PageValidationMsg;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.utils.CollectionUtils;
import ru.yandex.partner.jsonapi.crnk.block.BlockCrnkJsonFieldAliases;
import ru.yandex.partner.jsonapi.crnk.block.filter.provider.BlockTypeValuesProvider;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiField;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.crnk.filter.CrnkFilter;
import ru.yandex.partner.jsonapi.messages.block.RtbMsg;
import ru.yandex.partner.jsonapi.models.ApiModelPart;
import ru.yandex.partner.jsonapi.models.block.BlockFilterNameEnum;
import ru.yandex.partner.jsonapi.utils.function.BatchFunction;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;
import ru.yandex.partner.libs.i18n.MsgWithArgs;

import static ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRules.alwaysPermit;

public class ApiMobileBlockTypeModelPart<B extends BlockWithBlockType, P extends BasePage>
        implements ApiModelPart<B> {

    private final Set<MobileBlockType> supportedBlockTypes;
    private final ReachablePageService<B> reachablePageService;
    private final Supplier<CoreFilterNode<P>> authPageFilterProvider;
    private final MessageSourceAccessor translations;
    private final Class<P> pageClass;

    public ApiMobileBlockTypeModelPart(Set<MobileBlockType> supportedBlockTypes,
                                       ReachablePageService<B> reachablePageService,
                                       Supplier<CoreFilterNode<P>> authPageFilterProvider,
                                       MessageSource messageSource, Class<P> pageClass) {
        this.supportedBlockTypes = supportedBlockTypes;
        this.reachablePageService = reachablePageService;
        this.authPageFilterProvider = authPageFilterProvider;
        this.translations = new MessageSourceAccessor(messageSource);
        this.pageClass = pageClass;
    }

    @Override
    public List<ApiField<B>> getFields() {
        return List.of(
                ApiField.<B, String>convertibleApiField(String.class, B.BLOCK_TYPE)
                        .withAddFieldFunction(alwaysPermit())
                        .withAvailableFunction(alwaysPermit())
                        .withDefaultsFunction(getDefaultsFunction())
                        .build(BlockCrnkJsonFieldAliases.BLOCK_TYPE),
                ApiField.<B, String>apiField(String.class, BatchFunction.one(model ->
                                translations.getMessage(MobileBlockType.extract(model).getMsg()))
                        )
                        .withAvailableFunction(alwaysPermit())
                        .build(BlockCrnkJsonFieldAliases.BLOCK_TYPE_LABEL)
        );
    }

    private Function<QueryParamsContext<B>, Map<String, Object>> getDefaultsFunction() {
        return context -> {
            B block = context.getModelFromAttributes();
            Long pageId = block.getPageId();
            if (pageId == null) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS,
                        MsgWithArgs.of(BlockValidationMsg.PAGE_ID_EXPECTED, pageId));
            }
            Optional<P> page = reachablePageService.getReachablePages(
                    authPageFilterProvider.get().and(PageFilters.PAGE_ID.eq(pageId)),
                    pageClass
            ).stream().findAny();

            if (page.isEmpty()) {
                throw new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS,
                        MsgWithArgs.of(PageValidationMsg.PAGE_NOT_FOUND, pageId));
            }

            return supportedBlockTypes.stream()
                    .map(type -> Map.of(
                            "id", type.getLiteral(),
                            "label", translations.getMessage(type.getMsg())
                    ))
                    .collect(CollectionUtils.keyToList(BlockCrnkJsonFieldAliases.BLOCK_TYPE));
        };
    }

    @Override
    public Map<String, CrnkFilter<B, ?>> getFilters() {
        var list = List.<CrnkFilter<B, ?>>of(
                CrnkFilter.builder(BlockFilterNameEnum.BLOCK_TYPE)
                        .setLabel(RtbMsg.BLOCK_TYPE)
                        .setExposed(() -> true)
                        .toCrnkFilter(
                                BlockFilters.BLOCK_TYPE,
                                new BlockTypeValuesProvider(supportedBlockTypes)
                        )
        );

        return list.stream()
                .collect(Collectors.toMap(
                        CrnkFilter::getName,
                        baseBlockCrnkFilter -> baseBlockCrnkFilter
                ));
    }

}
