package ru.yandex.partner.jsonapi.models.block.rtb.mobile.external;

import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.dsp.service.DspService;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.jsonapi.crnk.exceptions.CrnkResponseStatusException;
import ru.yandex.partner.jsonapi.crnk.fields.ApiFieldsAccessRulesFunction;
import ru.yandex.partner.jsonapi.crnk.fields.QueryParamsContext;
import ru.yandex.partner.jsonapi.models.block.BlockRightNames;
import ru.yandex.partner.jsonapi.models.block.access.NotCustomBkDataAccessRuleFunction;
import ru.yandex.partner.jsonapi.models.block.parts.ApiDspsModelPart;
import ru.yandex.partner.libs.auth.model.UserAuthentication;
import ru.yandex.partner.libs.exceptions.HttpErrorStatusEnum;


public class ApiMobileRtbDspsModelPart extends ApiDspsModelPart<MobileRtbBlock> {

    private final DspService dspService;
    private final ReachablePageService<MobileRtbBlock> reachablePageService;

    public ApiMobileRtbDspsModelPart(String modelName, DspService dspService,
                               ReachablePageService<MobileRtbBlock> reachablePageService) {
        super(modelName);
        this.dspService = dspService;
        this.reachablePageService = reachablePageService;
    }

    @Override
    protected ApiFieldsAccessRulesFunction<MobileRtbBlock> isUnmoderatedRtbAuctionFunction() {
        return new ApiFieldsAccessRulesFunction<>((userAuthentication, rtbBlock) ->
                Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                MobileRtbBlock.CAMPAIGN
        );

    }

    @Override
    protected ApiFieldsAccessRulesFunction<MobileRtbBlock> isBlockDspsUnmoderatedAvailableFunction() {
        return new ApiFieldsAccessRulesFunction<>(
                (userAuthentication, rtbBlock) ->
                        userAuthentication.userHasRight(
                                BlockRightNames.VIEW_FIELD__UNMODERATED_DSPS.getFullRightName(getModelName()))
                                && rtbBlock.getCampaign() != null
                                && Boolean.TRUE.equals(rtbBlock.getCampaign().getUnmoderatedRtbAuction()),
                MobileRtbBlock.CAMPAIGN);
    }

    @Override
    public ApiFieldsAccessRulesFunction<MobileRtbBlock> isGodModeFunction() {
        return new NotCustomBkDataAccessRuleFunction<>();
    }

    @Override
    protected Function<QueryParamsContext<MobileRtbBlock>, Map<String, Object>> getDspsDefaultsFunction() {
        return context -> {
            MobileRtbBlock block = context.getModelFromAttributes();
            MobileAppSettings campaign = Optional.ofNullable(block.getPageId())
                    .map(pageId -> CoreFilterNode.eq(PageFilters.PAGE_ID, pageId))
                    .map(filter -> reachablePageService.getReachablePages(filter, MobileAppSettings.class))
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__PARAMS))
                    .stream().findFirst()
                    .orElseThrow(() -> new CrnkResponseStatusException(HttpErrorStatusEnum.ERROR__NOT_FOUND));
            block.setCampaign(campaign);

            UserAuthentication user = context.getUserAuthentication();
            boolean userHasRight = user.userHasRight(BlockRightNames.VIEW_FIELD__DSPS.getFullRightName(getModelName()));
            boolean canEditRichMedia = user.userHasRight(BlockRightNames.EDIT_FIELD__RICH_MEDIA
                    .getFullRightName(getModelName()));
            return dspService.getDefaults(block, userHasRight, campaign.getUnmoderatedRtbAuction(), canEditRichMedia);
        };
    }
}
